/*
    module/comedi_module.h
    header file for kernel-only structures, variables, and constants

    COMEDI - Linux Control and Measurement Device Interface
    Copyright (C) 1997-8 David A. Schleef <ds@stm.lbl.gov>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef _COMEDI_MODULE_H
#define _COMEDI_MODULE_H

#include <linux/version.h>
#include <linux/config.h>
#include <linux/kdev_t.h>
#include <config.h>
#include <linux/malloc.h>
#include <linux/errno.h>

#if LINUX_VERSION_CODE < 0x020000
#error kernel versions prior to 2.0 not supported
#else
#if LINUX_VERSION_CODE < 0x020100
#define LINUX_V20
#else
#define LINUX_V22
#endif
#endif

#include <comedi.h>
#include "version.h"


typedef struct comedi_device_struct comedi_device;
typedef struct comedi_subdevice_struct comedi_subdevice;

typedef int comedi_devinit(comedi_device *,comedi_devconfig *);

typedef struct comedi_bufll{
	struct comedi_bufll *next;
	int use_count;
	char stuff[0];
}comedi_bufll;

typedef struct{
	comedi_bufll *first;
	comedi_bufll *last;
	int in;
	int use_count;
	int num_pages;
}comedi_buf;

typedef struct{
	comedi_buf *buf;
	comedi_bufll *bufll;
	int offset;
}comedi_bufptr;

struct comedi_subdevice_struct{
	int type;
	int n_chan;
	int subdev_flags;
	int timer_type;
	int len_chanlist;		/* length of channel/gain list, if available */
	unsigned int maxbufsz;

	void *lock;
	void *busy;
	
	int io_bits;
	
	sampl_t maxdata;		/* if maxdata==0, use list */
	sampl_t *maxdata_list;		/* list is channel specific */
	
	unsigned int flags;
	unsigned int *flaglist;

	unsigned int range_type;
	unsigned int *range_type_list;
	
	unsigned int *chanlist;		/* driver-owned chanlist (not used) */
	
	comedi_trig	cur_trig;	/* current trig structure */
	
	sampl_t		*buf;		/* pre-allocated buffer */

	unsigned int buf_len;
	volatile unsigned int buf_int_ptr;	/* buffer marker for interrupt */
	unsigned int buf_user_ptr;	/* buffer marker for read() and write() */
	unsigned int cur_chan;		/* useless channel marker for interrupt */
	
	unsigned int *range_list;	/* is this necessary? */
	
	int (*trig[5])(comedi_device *,comedi_subdevice *,comedi_trig *);

	int (*cancel)(comedi_device *,comedi_subdevice *);

	int (*end_of_scan)(void *);
	int (*end_of_acquisition)(void *);

	void *eos_arg;
	void *eoa_arg;

	unsigned int state;
};

struct comedi_device_struct{
	int use_count;
	void *private;
	kdev_t minor;
	int (*rem)(comedi_device *);
	char *driver_name;
	char *board_name;
	int board;			/* this would be more useful as void* */

	int n_subdevices;
	comedi_subdevice *subdevices;
	int options[COMEDI_NDEVCONFOPTS];

	/* dumb */
	int iobase;
	int iosize;
	int irq;

	struct wait_queue *wait;

	/* still needed? */
	comedi_buf rdbuf;
	void (*poll)(comedi_device *);
	int polling;

};



extern comedi_device *comedi_devices;
extern comedi_devinit *comedi_devinits[];
extern const int comedi_maxinits;

/*
 * function prototypes
 */

void comedi_error(comedi_device *dev,const char *s);
void comedi_done(comedi_device *dev,comedi_subdevice *s);
void comedi_eos(comedi_device *dev,comedi_subdevice *s);

void init_polling(void);
void cleanup_polling(void);
void start_polling(comedi_device *);
void stop_polling(comedi_device *);

void comedi_proc_init(void);
void comedi_proc_cleanup(void);

/* range types */
/* autogenerated */

/* --BEGIN-RANGE-DEFS-- */
/* 	lib/range.def */
/* 	definition file for range information */
/*  */
/* 	COMEDI - Linux Control and Measurement Device Interface */
/* 	Copyright (C) 1998 David A. Schleef <ds@stm.lbl.gov> */
/*  */
/* 	Automatically generated from lib/range.def by scripts/util/mk_range */
/*  */
/*  general form for range.def file: */
/* 	* unique_name_for_this_range */
/* 		first_range_min		first_range_max		unit */
/* 		second_range_min	second_range_max	unit */
/*  etc... */
/*  */
/*  unit is either V, A, ext, or none.  Default is V. */
/*  unit 'ext' refers to external reference */
/*  */
/*  This file is _append only_.  Inserting range definitions breaks */
/*  binary compatibility. */
#define RANGE_bipolar10 __RANGE(0,1)
#define RANGE_bipolar5 __RANGE(1,1)
#define RANGE_bipolar2_5 __RANGE(2,1)
#define RANGE_unipolar10 __RANGE(3,1)
#define RANGE_unipolar5 __RANGE(4,1)
#define RANGE_unipolar2_5 __RANGE(5,1)
#define RANGE_dt282x_ai_lo_bipolar __RANGE(6,4)
#define RANGE_dt282x_ai_lo_unipolar __RANGE(10,4)
#define RANGE_dt282x_ai_5_bipolar __RANGE(14,4)
#define RANGE_dt282x_ai_5_unipolar __RANGE(18,4)
#define RANGE_dt282x_ai_hi_bipolar __RANGE(22,4)
#define RANGE_dt282x_ai_hi_unipolar __RANGE(26,4)
#define RANGE_ni_E_ai __RANGE(30,16)
#define RANGE_ni_E_ai_limited __RANGE(46,8)
#define RANGE_ni_E_ai_limited14 __RANGE(54,14)
#define RANGE_pcl711b_ai __RANGE(68,5)
#define RANGE_acl8112hg_ai __RANGE(73,12)
#define RANGE_acl8112dg_ai __RANGE(85,9)
#define RANGE_rti800_ai_10_bipolar __RANGE(94,4)
#define RANGE_rti800_ai_5_bipolar __RANGE(98,4)
#define RANGE_rti800_ai_10_unipolar __RANGE(102,4)
#define RANGE_das1601_ai_10_bipolar __RANGE(106,4)
#define RANGE_das1601_ai_10_unipolar __RANGE(110,4)
#define RANGE_das1602_ai_10_bipolar __RANGE(114,4)
#define RANGE_das1602_ai_10_unipolar __RANGE(118,4)
#define RANGE_unknown __RANGE(122,1)
#define RANGE_ni_E_ao_ext __RANGE(123,4)
#define RANGE_ni_E_ao __RANGE(127,2)
/* ---END-RANGE-DEFS--- */

/* extras */
/* someone should come up with a good way to merge these
   into range.def */

#define RANGE_digital			RANGE_unipolar5
#define RANGE_rti800_ao_10_bipolar		RANGE_bipolar10
#define RANGE_rti800_ao_10_unipolar		RANGE_unipolar10
#define RANGE_rti802_ao_10_bipolar		RANGE_bipolar10
#define RANGE_rti802_ao_10_unipolar		RANGE_unipolar10

#define RANGE_das1600_ao_5_bipolar		RANGE_bipolar5
#define RANGE_das1600_ao_10_bipolar		RANGE_bipolar10
#define RANGE_das1600_ao_extern_bipolar		0 /* XXX uhhh... */
#define RANGE_das1600_ao_5_unipolar		RANGE_bipolar5
#define RANGE_das1600_ao_10_unipolar		RANGE_bipolar10
#define RANGE_das1600_ao_extern_unipolar	0 /* XXX uhhh... */



/* timer types */
/* these *must* agree with lib/timer.c */

#define TIMER_dt282x			1
#define TIMER_dt2814			2
#define TIMER_atmio			3
#define TIMER_acl8112			4


/* some silly little inline functions */

static inline int alloc_subdevices(comedi_device *dev)
{
	int size=sizeof(comedi_subdevice)*dev->n_subdevices;

	dev->subdevices=kmalloc(size,GFP_KERNEL);
	if(!dev->subdevices)
		return -ENOMEM;
	memset(dev->subdevices,0,size);
	return 0;
}

static inline int alloc_private(comedi_device *dev,int size)
{
	dev->private=kmalloc(size,GFP_KERNEL);
	if(!dev->private)
		return -ENOMEM;
	memset(dev->private,0,size);
	return 0;
}

/* kernel compatibility */

#ifdef LINUX_V22

/* this is a useless abstraction that keeps several #ifdefs out of module.c */
#define minor_of_file(x)        MINOR((x)->f_dentry->d_inode->i_rdev)

#endif

#ifdef LINUX_V20

#define minor_of_file(x)        MINOR((x)->f_inode->i_rdev)

#define ioremap vremap
#define iounmap vfree

#define signal_pending(x)	(((x)->signal) & (~(x)->blocked))

#if CONFIG_EXPORT
extern struct symbol_table comedi_syms;
#endif

#include <asm/segment.h>

static inline int copy_to_user(void * to,const void *from,unsigned long n_bytes)
{
	int i;

	if((i=verify_area(VERIFY_WRITE,to,n_bytes)) != 0)
		return i;
	memcpy_tofs(to,from,n_bytes);
	return 0;
}

static inline int copy_from_user(void * to,const void *from,unsigned long n_bytes)
{
	int i;
	if((i=verify_area(VERIFY_READ,from,n_bytes))!=0)
		return i;
	memcpy_fromfs(to,from,n_bytes);
	return 0;
}

static inline int clear_user(void * mem,unsigned long len)
{
	char *cmem=mem;
	
	if(verify_area(VERIFY_WRITE,mem,len))
		return len;
	/* this is slow, but I'm lazy */
	while(len--){put_user(0,cmem);cmem++;}
	return 0;
}

#endif


#ifdef LINUX_V20
/* actually LINUX_VERSION_CODE < 0x020155 */

#define LINUX_old_PCI_compatibility

#endif


#endif /* _COMEDI_MODULE_H */




