/*
    module/ni-E.c
    Hardware driver for NI E series cards

    COMEDI - Linux Control and Measurement Device Interface
    Copyright (C) 1997-8 David A. Schleef <ds@stm.lbl.gov>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/*
	This file needs to be included by another file, e.g.,
	atmio-E.c.

	Interrupt support added by Truxton Fulton <trux@truxton.com>

	References for specifications:
	
	   321747b.pdf  Register Level Programmer Manual (obsolete)
	   321747c.pdf  Register Level Programmer Manual (new)
	   DAQ-STC reference manual

	Other possibly relevant info:
	
	   320517c.pdf  User manual (obsolete)
	   320517f.pdf  User manual (new)
	   320889a.pdf  delete
	   320906c.pdf  maximum signal ratings
	   321066a.pdf  about 16x
	   321791a.pdf  discontinuation of at-mio-16e-10 rev. c
	   321808a.pdf  about at-mio-16e-10 rev P
	   321837a.pdf  discontinuation of at-mio-16de-10 rev d
	   321838a.pdf  about at-mio-16de-10 rev N
	
	ISSUES:

	deal with at-mio-16de-10 revision D to N changes, etc.
	
*/

#include <8255.h>

#define static

/* reference: ground, common, differential, other */
static int ni_modebits1[4]={ 0x3000, 0x2000, 0x1000, 0 };
static int ni_modebits2[4]={ 0x3f, 0x3f, 0x37, 0x37 };

static int ni_gainlkup[][16]={
	{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
	{ 1, 2, 4, 7, 9, 10, 12, 15, 0,0,0,0,0,0,0,0 },
	{ 1, 2, 3, 4, 5, 6, 7, 9, 10, 11, 12, 13, 14, 15, 0,0 }
};

static int ni_range_lkup[]={
	RANGE_ni_E_ai,
	RANGE_ni_E_ai_limited,
	RANGE_ni_E_ai_limited14
};



static int ni_dio(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);
static int ni_rem(comedi_device *dev);
static int ni_read_eeprom(comedi_device *dev,int addr);

static int ni_eeprom(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);
static int ni_calib(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);


static void ni_handle_fifo_half_full(comedi_device *dev);
static void ni_handle_fifo_dregs(comedi_device *dev);

static int ni_8255_callback(int dir,int port,int data,void *arg);

#undef TIMEMARK
#undef DEBUG

#ifdef TIMEMARK
#include <linux/time.h>
#endif

#ifdef CONFIG_COMEDI_RT
static void ni_E_interrupt(void)
{
	/* XXX this is terrible, but it's a limitation of RTLinux */
	comedi_device *dev=ni_rt_interrupt_dev;
#else
static void ni_E_interrupt(int irq,void *d,struct pt_regs * regs)
{
	comedi_device *dev=d;
#endif
#ifdef TIMEMARK
	struct timeval tv;
#endif
	comedi_subdevice *s=dev->subdevices;
	int status;
	int ack=0;
	int wsave;

/*
    If you want to use windowed registers in an interrupt, it is
    important that you restore the window address register.  If
    you change certain modes, e.g., AI_Configuration_Start/End,
    you need to set up software flags for non-interrupt routines.
*/
	wsave=win_save();
	
	status=ni_readw(AI_Status_1);
	if(status&(AI_Overrun_St|AI_Overflow_St))
		printk("ni_E: overrun/overflow status=0x%04x\n",status);
	

#ifdef TIMEMARK
do_gettimeofday(&tv);
printk("ni-E %d.%06d status=0x%04x\n",tv.tv_sec,tv.tv_usec,status);
#endif

if(!dev->subdevices->cur_trig.data){
       printk("aiee! cur_trig.data got zapped!\n");
       comedi_done(dev,s);
       return;
}

	if(status&AI_SC_TC_St){
#ifdef DEBUG
printk("ni-E: SC_TC interrupt\n");
#endif
		ni_handle_fifo_dregs(dev);
		win_out(0x0000,Interrupt_A_Enable_Register);
		comedi_done(dev,s);

		ack|=AI_SC_TC_Interrupt_Ack;
	}
	if(status&AI_FIFO_Half_Full_St){
		ni_handle_fifo_half_full(dev);
	}
	if((dev->subdevices->cur_trig.flags&TRIG_WAKE_EOS) && !(status&AI_FIFO_Empty_St)){
		int new_cur_chan;

		ni_handle_fifo_dregs(dev);

		new_cur_chan=s->buf_int_ptr/s->cur_trig.n_chan;
		if(new_cur_chan>s->cur_chan){
#ifdef DEBUG
printk("buf_int_ptr=%d\n",s->buf_int_ptr);
#endif
			s->cur_chan=new_cur_chan;
			comedi_eos(dev,dev->subdevices+0);
		}
	}

	if(ack){
		/* the AI_Configuration_Start is required */

		win_out(AI_Configuration_Start,Joint_Reset_Register);
		ni_writew(ack,Interrupt_A_Ack_Register);
		win_out(AI_Configuration_End,Joint_Reset_Register);
	}

	win_restore(wsave);
}


static void ni_handle_fifo_half_full(comedi_device *dev)
{
	int i;
	comedi_subdevice *s=dev->subdevices+0;

	/*
	   if we got a fifo_half_full interrupt, we can transfer fifo/2
	   samples without checking the empty flag.  It doesn't matter if
	   we transfer the rest of the samples, the performance trade-off
	   is minimal (checking empty flag for a few samples vs. having
	   1% more interrupts.)  At really high speeds, it's better to
	   ignore them.

	*/
	
if(s->buf_int_ptr+boardtype.ai_fifo_depth/2 > s->buf_len){
       printk("comedi: ni-E: (bug) about to overrun buffer HF\n");
       s->buf_int_ptr=0;
}

	/*
	   This will warn us if the FIFO depth is documented wrong.
	 */
	for(i=0;i<boardtype.ai_fifo_depth/2-1;i++)
		s->cur_trig.data[s->buf_int_ptr++]=ni_readw(ADC_FIFO_Data_Register);
	if(ni_readw(AI_Status_1)&AI_FIFO_Empty_St)
		printk("comedi: ni-E: (bug) FIFO smaller than expected\n");
	s->cur_trig.data[s->buf_int_ptr++]=ni_readw(ADC_FIFO_Data_Register);

#if 0
	/* debug code to determine actual fifo depth */
	
	while(!(ni_readw(AI_Status_1)&AI_FIFO_Empty_St)){
		s->cur_trig.data[s->buf_int_ptr++]=ni_readw(ADC_FIFO_Data_Register);
		i++;
	}
	printk("comedi: ni-E: apparent FIFO half-depth %d\n",i);
#endif

}

/*
   Empties the AI fifo
*/
static void ni_handle_fifo_dregs(comedi_device *dev)
{
	comedi_subdevice *s=dev->subdevices+0;

	/*
	   Too bad NI didn't have the foresight to return a
	   "fifo empty" value if we read past the end of the
	   FIFO.  It would have made this procedure twice
	   as fast.

	   We can calculate how many samples to transfer.
	   This would save a lot of time.
	*/

	while(!(ni_readw(AI_Status_1)&AI_FIFO_Empty_St)){
if(s->buf_int_ptr >= s->buf_len){
	printk("comedi: ni-E: buffer overrun HF\n");
	ni_readw(ADC_FIFO_Data_Register);
}else{
		s->cur_trig.data[s->buf_int_ptr++]=ni_readw(ADC_FIFO_Data_Register);
}
	}
#if 0
	while(!(ni_readw(AI_Status_1)&AI_FIFO_Empty_St)){
		ni_readw(ADC_FIFO_Data_Register);
		s->buf_int_ptr++;
	}
#endif
}

/*
   used for both cancel ioctl and board initialization

   this is pretty harsh for a cancel, but it works...
 */
static int ni_ai_reset(comedi_device *dev,comedi_subdevice *s)
{
	win_out(0x0000,Interrupt_A_Enable_Register);

	win_out(AI_Reset,Joint_Reset_Register);

	win_out(1,ADC_FIFO_Clear);

	/* ai configuration */

	win_out(AI_Configuration_Start,Joint_Reset_Register);

	win_out(0x0000,AI_Command_1_Register); /* reset pulses */
	win_out(0x000d,AI_Mode_1_Register);
	win_out(0x0000,AI_Mode_2_Register);
#if 0
	win_out((1<<6)|0x0000,AI_Mode_3_Register); /* generate FIFO interrupts on half full */
#else
	win_out((0<<6)|0x0000,AI_Mode_3_Register); /* generate FIFO interrupts on non-empty */
#endif
	win_out(0xa4a0,AI_Personal_Register); /* ? */
	win_out(0x032e,AI_Output_Control_Register);
	win_out(0x0060,AI_Trigger_Select_Register); /* trigger source */

	/* this should be done in _ai_modeX() */
	win_out(0x29e0,AI_START_STOP_Select_Register);

	/* the following registers should not be changed:
		Clock_and_FOUT_Register
		AI_Mode_1_Register
		AI_Mode_3_Register
		AI_Personal_Register
		AI_Output_Control_Register
		AI_Trigger_Select_Register
	*/

	win_out(0x3f80,Interrupt_A_Ack_Register); /* clear interrupts */

	win_out(AI_Configuration_End,Joint_Reset_Register);

	return 0;
}

static void ni_load_channelgain_list(comedi_device *dev,unsigned int n_chan,unsigned int *list);


/*
	Mode 0 is immediate
*/
static int ni_ai_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	int i;
	int wsave;
	
	wsave=win_save();

	win_out(1,ADC_FIFO_Clear);

	/* interrupt on errors */
	win_out(0x0020,Interrupt_A_Enable_Register) ;
	
#if 0
	/* this function needs to be fixed for multiple channels */
	ni_load_channelgain_list(dev,it->n_chan,it->chanlist);
#else
	ni_load_channelgain_list(dev,it->n_chan,it->chanlist);
#endif

#if 0
	/* needs start configuration */
	win_out(AI_START_Edge|AI_START_Sync|
		AI_STOP_Select(19)|AI_STOP_Sync,
		AI_START_STOP_Select_Register);
#endif


	win_out(1,AI_Command_1_Register);
	
	for(i=0;i<40;i++){
		if(!(ni_readw(AI_Status_1)&AI_FIFO_Empty_St)){
			it->data[0]=ni_readw(ADC_FIFO_Data_Register);
			if(boardtype.adbits==12){
				if(CR_RANGE(it->chanlist[0])<8)
					it->data[0]^=0x800;
				it->data[0]&=0xfff;
			}else{
				if(CR_RANGE(it->chanlist[0])<8)
					it->data[0]^=0x8000;
				it->data[0]&=0xffff;
			}
			win_restore(wsave);
			return 1;
		}
		udelay(25);
	}
	printk("ni_E: timeout 2\n");
	win_restore(wsave);
	return -ETIME;
}

static int ni_ai_mode2(comedi_device *dev,comedi_subdevice *s, comedi_trig *it);
/*
	mode 1 is timed, one channel
*/
static int ni_ai_mode1(comedi_device *dev,comedi_subdevice *s, comedi_trig *it)
{
	it->trigvar1=10;

	return ni_ai_mode2(dev,s,it);

#if 0
	/* use FIFO interrupts to clean FIFO */
	devpriv->lastchan=it->chanlist[0];
	
	win_out(1,ADC_FIFO_Clear);
	win_out(1,Configuration_Memory_Clear);

/*
	mode=getparamval(dev->chinfo[it->chanlist[0]].paramlist,COMEDI_AREF);
*/
	mode=0;

	chan=ni_modebits1[mode]|(it->chanlist[0]&ni_modebits2[mode]);

	ni_writew(chan,Configuration_Memory_High);
	ni_writew(0x8000|devpriv->aip[it->chanlist[0]],
		Configuration_Memory_Low);
	
	/* start configuration */
	win_out(AI_Configuration_Start,Joint_Reset_Register);

	/* stage number of samples */
	win_out((it->n-1)>>16,AI_SC_Load_A_Registers);
	win_out((it->n-1)&0xffff,AI_SC_Load_A_Registers+1);
	
	/* load SC */
	win_out(0x20,AI_Command_1_Register);
	
	/*
	    AI_SI_Special_Trigger_Delay=0
	    AI_Pre_Trigger=0
	  AI_START_STOP_Select_Register:
	    AI_START_Polarity=0 (?)	rising edge
	    AI_START_Edge=1		edge triggered
	    AI_START_Sync=1 (?)		
	    AI_START_Select=0		SI_TC
	    AI_STOP_Polarity=0		rising edge
	    AI_STOP_Edge=0		level
	    AI_STOP_Sync=1		
	    AI_STOP_Select=19		external pin (configuration mem)
	*/
	win_out(AI_START_Edge|AI_START_Sync|
		AI_STOP_Select(19)|AI_STOP_Sync,
		AI_START_STOP_Select_Register);

	win_out((it->trigvar>>16),AI_SI_Load_A_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_A_Registers+1);
	/* AI_SI_Initial_Load_Source=A */
	win_out(0,AI_Mode_2_Register);
	/* load SI */
	win_out(0x200,AI_Command_1_Register);

	/* stage freq. counter into SI B */
	win_out((it->trigvar>>16),AI_SI_Load_B_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_B_Registers+1);

	win_out(1,AI_SI2_Load_A_Register); /* 0,0 does not work. */
	win_out(1,AI_SI2_Load_B_Register);

	/* AI_SI2_Reload_Mode = alternate */
	/* AI_SI2_Initial_Load_Source = A */
	win_out(0x0100,AI_Mode_2_Register);

	/* AI_SI2_Load */
	win_out(0x0800,AI_Command_1_Register);

	/* AI_SI_Initial_Load_Source=0
	   AI_SI_Reload_Mode(0)
	   AI_SI2_Reload_Mode = alternate, AI_SI2_Initial_Load_Source = B */
	win_out(0x0300,AI_Mode_2_Register);

	/* end configuration */
	win_out(AI_Configuration_End,Joint_Reset_Register);

	if(dev->irq){
		/* interrupt on FIFO, errors, SC_TC */
		win_out(0x00a1,Interrupt_A_Enable_Register) ;
	}else{
		/* interrupt on nothing */
		win_out(0x0000,Interrupt_A_Enable_Register) ;
	}
	
	/* AI_SI2_Arm, AI_SI_Arm, AI_DIV_Arm, AI_SC_Arm */
	win_out(0x1540,AI_Command_1_Register);

	/* AI_START1_Pulse */
	win_out(AI_START1_Pulse,AI_Command_2_Register);

	/*dev->busy[0]=it->job;*/
	if(!dev->irq){
		start_polling(dev);
	}
	return 0;
#endif
}

static void ni_load_channelgain_list(comedi_device *dev,unsigned int n_chan,unsigned int *list)
{
	unsigned int chan,range,aref;
	unsigned int i;
	unsigned int hi,lo;

	win_out(1,Configuration_Memory_Clear);

	for(i=0;i<n_chan;i++){
		chan=CR_CHAN(list[i]);
		range=CR_RANGE(list[i]);
		aref=CR_AREF(list[i]);
		
		/* fix the external/internal range differences */
		/* XXX not necessary if sample fixup not done in interrupt */
		range=ni_gainlkup[boardtype.gainlkup][range];
		list[i]=CR_PACK(chan,range,aref);

		hi=ni_modebits1[aref]|(chan&ni_modebits2[aref]);
		ni_writew(hi,Configuration_Memory_High);

		lo=((i==n_chan-1)?0x8000:0) | ((range&0x8)<<5) | (range&0x7);
		ni_writew(lo,Configuration_Memory_Low);
	}

	/* prime the channel/gain list */

	win_out(1,AI_Command_1_Register);
	for(i=0;i<40;i++){
		if(!(ni_readw(AI_Status_1)&AI_FIFO_Empty_St)){
			win_out(1,ADC_FIFO_Clear);
			return;
		}
		udelay(25);
	}
	printk("ni_E: timeout 1\n");
}


/*
	mode 2 is timed, multi-channel
*/
static int ni_ai_mode2(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	int wsave;

	wsave=win_save();

	win_out(1,ADC_FIFO_Clear);

	ni_load_channelgain_list(dev,it->n_chan,it->chanlist);
	
	/* start configuration */
	win_out(AI_Configuration_Start,Joint_Reset_Register);

	/* stage number of scans */
	win_out((it->n-1)>>16,AI_SC_Load_A_Registers);
	win_out((it->n-1)&0xffff,AI_SC_Load_A_Registers+1);
	
	/* load SC (Scan Count) */
	win_out(0x20,AI_Command_1_Register);
	
	/*
	    AI_SI_Special_Trigger_Delay=0
	    AI_Pre_Trigger=0
	  AI_START_STOP_Select_Register:
	    AI_START_Polarity=0 (?)	rising edge
	    AI_START_Edge=1		edge triggered
	    AI_START_Sync=1 (?)		
	    AI_START_Select=0		SI_TC
	    AI_STOP_Polarity=0		rising edge
	    AI_STOP_Edge=0		level
	    AI_STOP_Sync=1		
	    AI_STOP_Select=19		external pin (configuration mem)
	 */
	win_out(AI_START_Edge|AI_START_Sync|
		AI_STOP_Select(19)|AI_STOP_Sync,
		AI_START_STOP_Select_Register);

	win_out((it->trigvar>>16),AI_SI_Load_A_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_A_Registers+1);
	/* AI_SI_Initial_Load_Source=A */
	win_out(0,AI_Mode_2_Register);
	/* load SI */
	win_out(0x200,AI_Command_1_Register);

	/* stage freq. counter into SI B */
	win_out((it->trigvar>>16),AI_SI_Load_B_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_B_Registers+1);

	win_out(it->trigvar1,AI_SI2_Load_A_Register); /* 0,0 does not work. */
	win_out(it->trigvar1,AI_SI2_Load_B_Register);

	/* AI_SI2_Reload_Mode = alternate */
	/* AI_SI2_Initial_Load_Source = A */
	win_out(0x0100,AI_Mode_2_Register);

	/* AI_SI2_Load */
	win_out(0x0800,AI_Command_1_Register);

	/* AI_SI_Initial_Load_Source=0
	   AI_SI_Reload_Mode(0)
	   AI_SI2_Reload_Mode = alternate, AI_SI2_Initial_Load_Source = B */
	win_out(0x0300,AI_Mode_2_Register);

	if(dev->irq){
		int bits;

		/* interrupt on FIFO, errors, SC_TC */
		bits=0x00a1;

		if(it->flags&TRIG_WAKE_EOS){
			/*generate FIFO interrupts on non-empty */
			win_out(AI_FIFO_Mode_NE|0x0000,AI_Mode_3_Register);
#if 0
			bits|=AI_START_Interrupt_Enable|AI_STOP_Interrupt_Enable;
#endif
		}else{
			/*generate FIFO interrupts on non-empty */
			win_out(AI_FIFO_Mode_HF|0x0000,AI_Mode_3_Register);
		}
		win_out(0x3f80,Interrupt_A_Ack_Register); /* clear interrupts */

		win_out(bits,Interrupt_A_Enable_Register) ;
	}else{
		/* interrupt on nothing */
		win_out(0x0000,Interrupt_A_Enable_Register) ;

		/* XXX start polling if necessary */
	}

#ifdef DEBUG
printk("end config\n");
#endif
	/* end configuration */
	win_out(AI_Configuration_End,Joint_Reset_Register);
	
	/* AI_SI2_Arm, AI_SI_Arm, AI_DIV_Arm, AI_SC_Arm */
	win_out(0x1540,AI_Command_1_Register);

	/* AI_START1_Pulse */
	win_out(AI_START1_Pulse,AI_Command_2_Register);
#ifdef DEBUG
printk("START1 pulse\n");
#endif

	/* XXX hack alert */
	s->cur_chan=0;

	win_restore(wsave);
	return 0;
}

/*
	mode 4 is external trigger for scans, timer for samples
	in a scan
*/
static int ni_ai_mode4(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	int wsave;

	wsave=win_save();

	win_out(1,ADC_FIFO_Clear);

	ni_load_channelgain_list(dev,it->n_chan,it->chanlist);
	
	/* start configuration */
	win_out(AI_Configuration_Start,Joint_Reset_Register);

	/* stage number of scans */
	win_out((it->n-1)>>16,AI_SC_Load_A_Registers);
	win_out((it->n-1)&0xffff,AI_SC_Load_A_Registers+1);
	
	/* load SC (Scan Count) */
	win_out(0x20,AI_Command_1_Register);
	
	/*
	    AI_SI_Special_Trigger_Delay=0
	    AI_Pre_Trigger=0
	  AI_START_STOP_Select_Register:
	    AI_START_Polarity=0 (?)	rising edge
	    AI_START_Edge=1		edge triggered
	    AI_START_Sync=1 (?)		
	    AI_START_Select=1		PFI0
	    AI_STOP_Polarity=0		rising edge
	    AI_STOP_Edge=0		level
	    AI_STOP_Sync=1		
	    AI_STOP_Select=19		external pin (configuration mem)
	*/
	win_out(AI_START_Edge|AI_START_Sync|AI_START_Select(1)|
		AI_STOP_Select(19)|AI_STOP_Sync,
		AI_START_STOP_Select_Register);
	
#if 0
	win_out((it->trigvar>>16),AI_SI_Load_A_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_A_Registers+1);
	/* AI_SI_Initial_Load_Source=A */
	win_out(0,AI_Mode_2_Register);
	/* load SI */
	win_out(0x200,AI_Command_1_Register);

	/* stage freq. counter into SI B */
	win_out((it->trigvar>>16),AI_SI_Load_B_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_B_Registers+1);
#endif

	win_out(it->trigvar1,AI_SI2_Load_A_Register); /* 0,0 does not work. */
	win_out(it->trigvar1,AI_SI2_Load_B_Register);

	/* AI_SI2_Reload_Mode = alternate */
	/* AI_SI2_Initial_Load_Source = A */
	win_out(0x0100,AI_Mode_2_Register);

	/* AI_SI2_Load */
	win_out(0x0800,AI_Command_1_Register);

	/* AI_SI_Initial_Load_Source=0
	   AI_SI_Reload_Mode(0)
	   AI_SI2_Reload_Mode = alternate, AI_SI2_Initial_Load_Source = B */
	win_out(0x0300,AI_Mode_2_Register);

	if(dev->irq){
		int bits;

		/* interrupt on FIFO, errors, SC_TC */
		bits=0x00a1;

		if(it->flags&TRIG_WAKE_EOS){
			/* generate FIFO interrupts on non-empty */
			win_out(AI_SI2_Source_Select|AI_FIFO_Mode_NE|0x0000,AI_Mode_3_Register);
		}else{
			/* generate FIFO interrupts on half-full */
			win_out(AI_SI2_Source_Select|AI_FIFO_Mode_HF|0x0000,AI_Mode_3_Register);
		}

		/* clear interrupts */
		win_out(0x3f80,Interrupt_A_Ack_Register);

		win_out(bits,Interrupt_A_Enable_Register) ;
	}else{
		/* interrupt on nothing */
		win_out(0x0000,Interrupt_A_Enable_Register) ;

		/* XXX start polling if necessary */
	}
	
	/* end configuration */
	win_out(AI_Configuration_End,Joint_Reset_Register);

	/* AI_SI2_Arm, AI_DIV_Arm, AI_SC_Arm */
	win_out(0x1500,AI_Command_1_Register);

	/* AI_START1_Pulse */
	win_out(AI_START1_Pulse,AI_Command_2_Register);

	/* XXX hack alert */
	s->cur_chan=0;

	win_restore(wsave);
	return 0;
}


static int ni_ao_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);

#if 0
/*
	analog output switch
*/
static int ni_ao(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	switch(it->mode){
	case 0:
		return ni_ao_mode0(dev,s,it);
	default:
		return -EINVAL;
	}
}
#endif


static int ni_ao_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	unsigned int data;
	unsigned int conf;
	unsigned int chan;
	unsigned int range;
	
	data=it->data[0];
	chan=CR_CHAN(it->chanlist[0]);

	conf=chan<<8;
	
	/* XXX check range with current range in flaglist[chan] */
	/* should update calibration if range changes (ick) */

	range = CR_RANGE(it->chanlist[0]);
	conf |= (range&1);
	conf |= (range&2)<<1;
	
	/* not all boards can deglitch, but this shouldn't hurt */
	if(it->flags & TRIG_DEGLITCH)
		conf |= 2;

	/* analog reference */
	/* AREF_OTHER connects AO ground to AI ground, i think */
	conf |= (CR_AREF(it->chanlist[0])==AREF_OTHER)? 8 : 0;

	ni_writew(conf,AO_Configuration);

	if(range&1)data^=0x800;
	
	ni_writew(data,(chan)? DAC1_Direct_Data : DAC0_Direct_Data);
	
	return 1;
}


/*
	digital i/o

	assume digital output for now
*/
static int ni_dio(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	int data,mask,data_in;
	int i;
	int temp;

	/* rt stuff */
	temp=win_save();

	if(it->flags & TRIG_CONFIG){
		data=s->io_bits;
		for(i=0;i<it->n_chan;i++){
			mask=1<<CR_CHAN(it->chanlist[i]);
			data&= ~mask;
			if(it->data[i])
				data|=mask;
		}
		s->io_bits=data;
		win_out(s->io_bits,DIO_Control_Register);
	}else{
		data=devpriv->last_do;
		data_in=win_in(DIO_Input_Register);
		for(i=0;i<it->n_chan;i++){
			mask=1<<CR_CHAN(it->chanlist[i]);
			if(s->io_bits & mask){
				/* output bit */
				data&= ~mask;
				if(it->data[i])
					data|=mask;
			}else{
				/* input bit */
				it->data[i]=(data_in&mask)?1:0;
			}
		}
		devpriv->last_do=data;

		win_out(data,DIO_Output_Register);
	}

	win_restore(temp);

	return i;
}

#if 0
static int ni_dsp(comedi_device *dev,comedi_param *it)
{
	switch(it->pnum){
	case COMEDI_IOBITS:
		if(it->pval&~0xff)return -EINVAL;
		devpriv->dio=it->pval;
		win_out(devpriv->dio,DIO_Control_Register);
		changeparam(&dev->chinfo[it->chan].paramlist,COMEDI_IOBITS,0,
			it->pval);
		return 0;
	default:
		return -EINVAL;
	}
}
#endif

/*
   XXX this assumes that if there isn't anything in the fifo,
   then we can turn off polling.  this is *not* a good idea,
   but it will do for now.
   Note: there is always the possibility that the poll
   routine will steal a sample from a soft triggered AI.
*/

#if 0
static void ni_E_poll(comedi_device *dev)
{
	sampl_t data;
	int wake=0;
	int i;
	int status;
#ifdef DEBUG
	int n=0;
#endif
	
	status=ni_readw(AI_Status_1);
#ifdef DEBUG
	printk("ni_E: polling... jiffies=%d status=0x%04x ",jiffies,status);
#endif

	if(status&AI_FIFO_Half_Full_St){
#ifdef DEBUG
		printk("fifo half full ");
		n+=boardtype.ai_fifo_depth/2;
#endif
		for(i=0;i<boardtype.ai_fifo_depth/2;i++){
			data=ni_readw(ADC_FIFO_Data_Register);
			if(!(devpriv->aip[devpriv->lastchan]&0x100))
				data^=0x800;
			data&=0xfff; /* incompatible with 16 bit cards */
			report_sample(dev,data);
		}
		wake=1;
	}
	while(!(ni_readw(AI_Status_1)&AI_FIFO_Empty_St)){
		data=ni_readw(ADC_FIFO_Data_Register);
		if(!(devpriv->aip[devpriv->lastchan]&0x100))
			data^=0x800;
		data&=0xfff; /* incompatible with 16 bit cards */
		report_sample(dev,data);
		wake=1;
#ifdef DEBUG
		n++;
#endif
	}
#ifdef DEBUG
	printk("samples=%d\n",n);
#endif

	if(wake)comedi_done(dev,dev->subdevices+0);
	else{
		stop_polling(dev);
		/*dev->busy[0]=0;*/
	}
}
#endif


static int ni_E_init(comedi_device *dev,comedi_devconfig *it)
{
	comedi_subdevice *s;
	
	dev->n_subdevices=7;
	
	dev->subdevices=kmalloc(dev->n_subdevices*sizeof(comedi_subdevice),GFP_KERNEL);
	if(!dev->subdevices){
		return -ENOMEM;
	}
	memset(dev->subdevices,0,dev->n_subdevices*sizeof(comedi_subdevice));
	
	/* analog input subdevice */

	s=dev->subdevices+0;
	s->type=COMEDI_SUBD_AI;
	s->subdev_flags=SDF_READABLE|SDF_GROUND|SDF_COMMON|SDF_DIFF|SDF_OTHER;
#ifdef CONFIG_COMEDI_RT
	s->subdev_flags|=SDF_RT;
#endif
#if 0
	s->subdev_flags|=SDF_DITHER;
#endif
	s->n_chan=boardtype.n_adchan;
	s->len_chanlist=512;	/* XXX is this the same for PCI-MIO ? */
	s->maxdata=(1<<boardtype.adbits)-1;
	s->timer_type=TIMER_atmio;
	s->range_type=ni_range_lkup[boardtype.gainlkup];
	s->trig[0]=ni_ai_mode0;
	s->trig[1]=ni_ai_mode1;		/* not true mode 1 */
	s->trig[2]=ni_ai_mode2;
	s->trig[4]=ni_ai_mode4;
	s->cancel=ni_ai_reset;
	
	/* analog output subdevice */
	/* XXX what about boards without ao? */

	s=dev->subdevices+1;
	s->type=COMEDI_SUBD_AO;
	s->subdev_flags=SDF_WRITEABLE|SDF_RT|SDF_DEGLITCH|SDF_GROUND|SDF_OTHER;
	s->n_chan=2;
	s->maxdata=(1<<12)-1;
	s->timer_type=TIMER_atmio;
	s->range_type=RANGE_ni_E_ao_ext;	/* XXX wrong for some boards */
	s->trig[0]=ni_ao_mode0;
	
	/* digital i/o subdevice */
	
	s=dev->subdevices+2;
	s->type=COMEDI_SUBD_DIO;
	s->subdev_flags=SDF_WRITEABLE|SDF_READABLE|SDF_RT;
	s->n_chan=8;
	s->maxdata=1;
	s->range_type=RANGE_digital;
	s->io_bits=0;		/* all bits input */
	s->trig[0]=ni_dio;

	/* dio setup */
	win_out(s->io_bits,DIO_Control_Register);
	
	/* 8255 device */
	s=dev->subdevices+3;
	if(boardtype.has_8255){
		subdev_8255_init(dev,s,ni_8255_callback,dev);
	}else{
		s->type=COMEDI_SUBD_UNUSED;
		s->trig[0]=NULL;
	}
	/* XXX */
	
	/* timer/counter device */
	s=dev->subdevices+4;
	s->type=COMEDI_SUBD_UNUSED;
	s->trig[0]=NULL;
	/* XXX */
	
	/* calibration subdevice -- ai and ao */
	s=dev->subdevices+5;
	s->type=COMEDI_SUBD_CALIB;
	s->subdev_flags=SDF_WRITEABLE|SDF_INTERNAL;
	s->n_chan=(boardtype.caldactype==1)?12:9;
	s->maxdata=0xfff;
	s->trig[0]=ni_calib;
	
	/* EEPROM */
	s=dev->subdevices+6;
	s->type=COMEDI_SUBD_MEMORY;
	s->subdev_flags=SDF_READABLE|SDF_INTERNAL;
	s->n_chan=512;
	s->maxdata=0xff;
	s->trig[0]=ni_eeprom;
	
	/* ai configuration */
	ni_ai_reset(dev,dev->subdevices+0);
	win_out(0x1ba0,Clock_and_FOUT_Register);


	/* analog output configuration */
	
	devpriv->ao0p=0x0000;
	ni_writew(devpriv->ao0p,AO_Configuration);
	devpriv->ao1p=0x0100;
	ni_writew(devpriv->ao1p,AO_Configuration);
	win_out(AO_Configuration_Start,Joint_Reset_Register);
	win_out(0x2000,AO_Command_1_Register);
	win_out(0,Interrupt_B_Enable_Register);
	win_out(0x0010,AO_Personal_Register);
	ni_writew(0x3f98,Interrupt_B_Ack);
	win_out(0x1430,AO_Personal_Register);
	win_out(0,AO_Output_Control_Register);
	win_out(0,AO_Start_Select_Register);
	win_out(0,AO_Command_1_Register);

        if(dev->irq){
                win_out((IRQ_POLARITY<<0) |  /* polarity : active high */
                        (0<<1) |  /* no interrupt on 3 pins */
                        (1<<11) |  /* enable int A */
                        (interrupt_pin(dev->irq)<<8),  /* interrupt output pin */
                        Interrupt_Control_Register
                );
        }

	
	dev->rem=ni_rem;

	printk("\n");

	return 1;
}



/* called when driver is removed */
static int ni_rem(comedi_device *dev)
{
	printk("comedi%d: ni_E: remove\n",dev->minor);
	
	/* board-specific cleanup */
	ni_E_rem_stage2(dev);
	
	return 0;
}


/*
	presents the EEPROM as a subdevice
*/

static int ni_eeprom(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	if(it->mode != 0)
		return -EINVAL;

	it->data[0]=ni_read_eeprom(dev,CR_CHAN(it->chanlist[0]));

	return 1;
}

/*
	reads bytes out of eeprom
*/

static int ni_read_eeprom(comedi_device *dev,int addr)
{
	int bit;
	int bitstring;

	bitstring=0x0300|((addr&0x100)<<3)|(addr&0xff);
	ni_writeb_p(0x04,Serial_Command);
	for(bit=0x8000;bit;bit>>=1){
		ni_writeb_p(0x04|((bit&bitstring)?0x02:0),Serial_Command);
		ni_writeb_p(0x05|((bit&bitstring)?0x02:0),Serial_Command);
	}
	bitstring=0;
	for(bit=0x80;bit;bit>>=1){
		ni_writeb_p(0x04,Serial_Command);
		ni_writeb_p(0x05,Serial_Command);
		bitstring|=((ni_readb_p(Status)&0x01)?bit:0);
	}
	ni_writeb_p(0x00,Serial_Command);
	
	return bitstring;
}

static void ni_write_caldac(comedi_device *dev,int addr,int val);
/*
	calibration subdevice
*/
static int ni_calib(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	if(it->mode != 0)
		return -EINVAL;

	ni_write_caldac(dev,CR_CHAN(it->chanlist[0]),it->data[0]);

	return 1;
}

static void ni_write_caldac(comedi_device *dev,int addr,int val)
{
	int loadbit,bits,bit,bitstring;
	
	switch(boardtype.caldactype){
	case 1:
		/*
		   MB 88341
		   Note that address bits are reversed.  Thanks to
		   Ingo Keen for noticing this.

		   Note also that the 88341 expects address values from
		   1-12, whereas we use channel numbers 0-11.  The NI
		   docs use 1-12, also, so be careful here.
		*/
		addr++;
		bits=12;
		bitstring=((addr&0x1)<<11) |
			  ((addr&0x2)<<9)  |
			  ((addr&0x4)<<7)  |
			  ((addr&0x8)<<5)  |
			  (val&0xff);
		loadbit=0x08;
		break;
	case 2:
		if(addr==8){
			/*
			   DAC 8043
			*/
			bits=12;
			bitstring=val&0xfff;
			loadbit=0x10;
		}else{
			/*
			   DAC 8800
			*/
			bits=11;
			bitstring=((addr&0x7)<<8)|(val&0xff);
			loadbit=0x08;
		}
		break;
	default:
		return;
	}

	for(bit=1<<(bits-1);bit;bit>>=1){
		ni_writeb(((bit&bitstring)?0x02:0),Serial_Command);
		ni_writeb(1|((bit&bitstring)?0x02:0),Serial_Command);
	}
	ni_writeb(loadbit,Serial_Command);
	ni_writeb(0,Serial_Command);
}



/*
 *
 *  Programmable Function Inputs
 *
 */

#if 0

static void pfi_setup(comedi_device *dev)
{
	/* currently, we don't output any signals, thus, all
	   the PFI's are input */
	
	win_out(IO_Bidirection_Pin_Register,0);
}



/*
 *
 *  General Purpose Counter/Timer section
 *
 */


/* event counting */

int gpct_start(comedi_device *dev,int chan)
{
	/* select load source */
	Gi_Load_Source_Select = 0;
	
	/* load initial counter value */
	Gi_Load_A = 0;
	
	/* strobe */
	Gi_Load=1;

/* command reg */
	Gi_Up_Down = 1; /* up counting */
	Gi_Bank_Switch_Enable = 0;
	Gi_Bank_Switch_Mode = 0;

/* input select reg */
	Gi_Source_Select = PFIn;
	Gi_Source_Polarity = 0; /* rising edges */
	Gi_Gate_Select = 31; /* logic low */
	Gi_OR_Gate = 0;
	Gi_Output_Polarity = 1; /* active low */
	Gi_Gate_Select_Load_Source = 0;

/* mode reg */
	Gi_Gate_Polarity = 0; /* disable inversion */
	Gi_Loading_On_Gate = 0;
	Gi_Output_Mode = 1; /* one clock cycle output */
	Gi_Loading_On_TC = 0;
	Gi_Gating_Mode = 1;
	Gi_Gate_On_Both_Edges = 0;
	Gi_Trigger_Mode_For_Edge_Gate = 2;
	Gi_Stop_Mode = 0;
	Gi_Counting_Once = 0;
	
	
/* int enable reg -- ignore */
	
/*
	Gi_TC_Interrupt_Enable = 0;
	Gi_Gate_Interrupt_Enable = 0;
*/
}

static int gpct_sp(comedi_device *dev,comedi_param *it)
{
	switch(it->pnum){
	case COMEDI_CTR_INPUT:
	{
		/* which input source? */
		int pval=it->pval;
		
		if(pval<0 || pval>=10)return -EINVAL;
		changeparam(&dev->chinfo[it->chan].paramlist,COMEDI_CTR_INPUT,
			0,pval);
		/* XXX set it */
		return 0;
	}
	case COMEDI_CTR_POLARITY:
		/* count rising or falling edges? */
		if(it->pval<0 || it->pval>=4)return -EINVAL;
		changeparam(&dev->chinfo[it->chan].paramlist,COMEDI_AREF,
			0,it->pval);
		/* XXX set it */
		return 0;
	case COMEDI_COUNT:
		changeparam(&dev->chinfo[it->chan].paramlist,COMEDI_COUNT,
			0,it->pval);  /* irrelevant... */
		devpriv->gpct[it->chan-boardtype->counteroffset]=it->pval;
		return 0;
	}
	return -EINVAL;
}

#endif

static int ni_8255_callback(int dir,int port,int data,void *arg)
{
	comedi_device *dev=arg;

	if(dir){
		ni_writeb(25+2*port,data);
		return 0;
	}else{
		return ni_readb(25+2*port);
	}
}

