/*
    module/pcimio-E.c
    Hardware driver for NI PCI-MIO E series cards

    COMEDI - Linux Control and Measurement Device Interface
    Copyright (C) 1997-8 David A. Schleef <ds@stm.lbl.gov>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/*
	The PCI-MIO E series driver was originally written by
	Tomasz Motylewski <...>, and ported to comedi by ds.


	References for specifications:
	
	   321747b.pdf  Register Level Programmer Manual (obsolete)
	   321747c.pdf  Register Level Programmer Manual (new)
	   DAQ-STC reference manual

	Other possibly relevant info:
	
	   320517c.pdf  User manual (obsolete)
	   320517f.pdf  User manual (new)
	   320889a.pdf  delete
	   320906c.pdf  maximum signal ratings
	   321066a.pdf  about 16x
	   321791a.pdf  discontinuation of at-mio-16e-10 rev. c
	   321808a.pdf  about at-mio-16e-10 rev P
	   321837a.pdf  discontinuation of at-mio-16de-10 rev d
	   321838a.pdf  about at-mio-16de-10 rev N
	
	ISSUES:

	need to deal with external reference for DAC, and other DAC
	properties in board properties
	
	deal with at-mio-16de-10 revision D to N changes, etc.
	
	need to add other CALDAC type
	
	need to slow down DAC loading.  I don't trust NI's claim that
	two writes to the PCI bus slows IO enough.  I would prefer to
	use udelay().  Timing specs: (clock)
		AD8522		30ns
		DAC8043		120ns
		DAC8800		60ns
		MB88341		?

*/

#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/mm.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <asm/io.h>
#include <linux/malloc.h>
#include <comedi_module.h>
#include <ni_stc.h>
#ifdef LINUX_old_PCI_compatibility
#include <linux/bios32.h>
#endif

#undef DEBUG
#define PCI_DEBUG


#define PCIMIO 1
#undef ATMIO

#ifndef CONFIG_PCI
#error CONFIG_PCI (in Linux kernel config) is not set.  It is necessary the pcimio-E driver.
#endif

/*
 *  PCI specific setup
 */


#define PCI_VENDOR_ID_NATINST		0x1093

#define PCI_MITE_SIZE		4096
#define PCI_DAQ_SIZE		4096


static ni_board ni_boards[]={
	{ 0x0162, "pci-mio-16xe-50",
		16,16,0xffff,0x8000,
		2048,1,1,
		2,12,0xfff,0x800,
		0,1, /* only bipolar */
		-1,-1,0 },
	{ 0x1170, "pci-mio-16xe-10",
		16,16,0xffff,0x8000,
		512,1,2,
		2,16,0xffff,0x8000,
		2048,0,
		-1,-1,0 },
	{ 0x1180, "pci-mio-16e-1",
		16,12,0xfff,0x800,
		512,0,0,
		2,12,0xfff,0x800,
		2048,0,
		-1,-1,0 },
	{ 0x1190, "pci-mio-16e-4",
		16,12,0xfff,0x800,
		512,0,0,
		2,12,0xfff,0x800,
		512,0,
		-1,-1,0 },
	{ 0x1330, "pci-6031e",
		64,16,0xffff,0x8000,
		512,1,2,
		2,16,0xffff,0x8000,
		2048,0,
		-1,-1,0 },
	{ 0x0000, "pci-6032e",
		16,16,0xffff,0x8000,
		512,1,2,
		0,0,0,0,
		0,0,
		-1,-1,0 },
	{ 0x0000, "pci-6033e",
		64,16,0xffff,0x8000,
		512,1,2,
		0,0,0,0,
		0,0,
		-1,-1,0 },
	{ 0x0000, "pci-6071e",
		64,16,0xffff,0x8000,
		-1,-1,-1,
		-1,-1,-1,-1,
		2048,0,
		-1,-1,0 },
	{ 0x0000, "pci-6110e" },
	{ 0x0000, "pci-6111e" },
};

#define serial_number_of_board(a)	0

comedi_devinit pcimio_init;


/* How we access registers */


#define ni_writew(a,b)		(writew((a),dev->iobase+(b)))
#define ni_readw(a)		(readw(dev->iobase+(a)))
#define ni_writeb(a,b)		(writeb((a),dev->iobase+(b)))
#define ni_readb(a)		(readb(dev->iobase+(a)))
#define ni_writeb_p(a,b)	(ni_writeb(a,b),ni_writeb(a,b))
#define ni_readb_p(a)		(ni_readb(a),ni_readb(a))


/*
 * this is how we access windowed registers
 * 
 * from a driver perspective, using windowed registers
 * on the PCI-MIO is really dumb.  I'd bet that the
 * registers can actually be acessed without windowing...
 * must try this sometime...
 */

#define win_out(a,b) (ni_writew((b),Window_Address),ni_writew((a),Window_Data))
#define win_in(b) (ni_writew((b),Window_Address),ni_readw(Window_Data))
#define win_save() (ni_readw(Window_Address))
#define win_restore(a) (ni_writew((a),Window_Address))

/*
   If interrupts _still_ don't work, play with the
   following two values.
 */
#define interrupt_pin(a)	0
#define IRQ_POLARITY 1

#ifdef CONFIG_COMEDI_RT
static void *ni_rt_interrupt_dev;
#endif


typedef struct{
	int board;
	int dio;
	int ao0p,ao1p;
	int aip[64];
	int lastchan;
	int last_do;
#ifdef LINUX_old_PCI_compatibility
	unsigned char pci_bus;
	unsigned char pci_device_fn;
#else
	struct pci_dev *pci_device;
#endif
	void *mite_io_addr;
	void *daq_io_addr;
	unsigned long daq_phys_addr;
	unsigned long mite_phys_addr;
}ni_private;
#define devpriv ((ni_private *)dev->private)

static pcimio_E_rem(comedi_device *dev);
#define ni_E_rem_stage2 pcimio_E_rem


#include "ni-E.c"


#ifdef LINUX_old_PCI_compatibility
static int ni_pci_find_device(unsigned char *pci_bus,unsigned char *pci_device_fn,int *board);
#else
static struct pci_dev * ni_pci_find_device(struct pci_dev *from,int *board);
#endif
static int init_stage2(comedi_device *dev,comedi_devconfig *it);
static int setup_mite(comedi_device *dev);
static void unset_mite(comedi_device *dev);


/* cleans up allocated resources */
int pcimio_E_free(comedi_device *dev)
{
	if(dev->private){
		unset_mite(dev);
	
		kfree(dev->private);
	}
	
	if(dev->irq){
#ifdef CONFIG_COMEDI_RT
		free_RTirq(dev->irq);
#else
		free_irq(dev->irq,dev);
#endif
	}

	return 0;
}

/* this gets called when the driver is removed */
static int pcimio_E_rem(comedi_device *dev)
{
	return pcimio_E_free(dev);
}

int pcimio_E_init(comedi_device *dev,comedi_devconfig *it)
{
	int		ret;
	
	if(strcmp("pcimio-E",it->board_name))
		return 0;
	
	if( !(ret=init_stage2(dev,it)) )
		return 1;
	
	pcimio_E_free(dev);
	
	return ret;
}



#ifdef LINUX_old_PCI_compatibility
/* old PCI stuff */
/* routines for the old PCI code (before 2.1.55) */

static int init_stage2(comedi_device *dev,comedi_devconfig *it)
{
	unsigned char	pci_bus=0;
	unsigned char	pci_dev_fn=0;
	int		board=0;
	int		ret;
	unsigned char	irq=0;
	
	
	printk("comedi%d: pcimio-E:",dev->minor);
	
	if(!pcibios_present()){
		printk(" no PCI bus\n");
		return -EINVAL;
	}
	
	if(it->options[0]){
		printk(" selection of board by serial number not (yet) supported\n");
		return -EINVAL;
	}
		

	if(!ni_pci_find_device(&pci_bus,&pci_dev_fn,&board)){
		printk(" no board found\n");
		return -EINVAL;
	}

	printk(" %s",ni_boards[board].name);
	dev->board_name=ni_boards[board].name;
	dev->driver_name="pcimio-E";

	dev->private=kmalloc(sizeof(ni_private),GFP_KERNEL);
	if(!dev->private){
		printk(" no memory\n");
		return -ENOMEM;
	}
	memset(dev->private,0,sizeof(ni_private));
	
	devpriv->board=board;
	devpriv->pci_bus=pci_bus;
	devpriv->pci_device_fn=pci_dev_fn;
	
	setup_mite(dev);
	
        /* irq stuff */

	pcibios_read_config_byte(pci_bus,pci_dev_fn,PCI_INTERRUPT_LINE,&irq);
        if(irq==0){
		printk(" unknown irq (bad)\n");
		return -EINVAL;
	}
        printk(" ( irq = %d )",irq);
#ifdef CONFIG_COMEDI_RT
        if( (ret=request_RTirq(irq,ni_E_interrupt))<0 ){
		ni_rt_interrupt_dev=dev;
		printk(" irq not available\n");
		return -EINVAL;
	}
#else
        if( (ret=request_irq(irq,ni_E_interrupt,SA_INTERRUPT,"pcimio-E",dev))<0 ){
                printk(" irq not available\n");
                return -EINVAL;
        }
#endif
        dev->irq=irq;

	if( (ret=ni_E_init(dev,it))<0 ){
		
		return ret;
	}
	
	return 0;
}

static int
ni_pci_find_device(unsigned char *pci_bus,unsigned char *pci_dev_fn,int *board)
{
	int i;
	
	for(i=0;i<n_ni_boards;i++){
		if(pcibios_find_device(PCI_VENDOR_ID_NATINST,
				ni_boards[i].device_id,
				0,  /* ? */
				pci_bus,
				pci_dev_fn) == PCIBIOS_SUCCESSFUL)
		{
			*board=i;
			return 1;
		}
	}
	return 0;
}


static int setup_mite(comedi_device *dev)
{
	unsigned long			offset;
	u32				addr;

	pcibios_read_config_dword(devpriv->pci_bus,devpriv->pci_device_fn,PCI_BASE_ADDRESS_1,&addr);
	devpriv->daq_phys_addr=addr;
	offset = devpriv->daq_phys_addr & ~PAGE_MASK;
	devpriv->daq_io_addr = ioremap(devpriv->daq_phys_addr & PAGE_MASK, PCI_DAQ_SIZE + offset )
		+ offset;
#ifdef PCI_DEBUG
	printk("DAQ:0x%08lx mapped to %p, ",devpriv->daq_phys_addr,devpriv->daq_io_addr);
#endif

	pcibios_read_config_dword(devpriv->pci_bus,devpriv->pci_device_fn,PCI_BASE_ADDRESS_0,&addr);
	devpriv->mite_phys_addr=addr;
	offset = devpriv->mite_phys_addr & ~PAGE_MASK;
	devpriv->mite_io_addr = ioremap(devpriv->mite_phys_addr & PAGE_MASK, PCI_MITE_SIZE + offset )
		+ offset;
#ifdef PCI_DEBUG
	printk("MITE:0x%08lx mapped to %p ",devpriv->mite_phys_addr,devpriv->mite_io_addr);
#endif
	
	/* XXX don't know what the 0xc0 and 0x80 mean */
	writel(devpriv->daq_phys_addr | 0x80 , devpriv->mite_io_addr + 0xc0 );
	
	dev->iobase = (int)devpriv->daq_io_addr;
	
	return 0;
}





#else

/* routines for the new PCI code (after 2.1.55) */



static int init_stage2(comedi_device *dev,comedi_devconfig *it)
{
	struct pci_dev	*pci_device=NULL;
	int		board=0;
	int		ret;
	int		irq;
	int		serial_number=0;
	
	
	printk("comedi%d: pcimio-E:",dev->minor);
	
	if(!pci_present()){
		printk(" no PCI bus\n");
		return -EINVAL;
	}
	
	if(it->options[0]){
		printk(" selection of board by serial number not (yet) supported\n");
		return -EINVAL;
	}
		
	while( (pci_device=ni_pci_find_device(pci_device,&board)) ){
		if(!serial_number || serial_number==serial_number_of_board(dev))
			break;
	}
	if(!pci_device){
		printk(" no board found\n");
		return -EINVAL;
	}
	devpriv->pci_device=pci_device;

	printk(" %s",ni_boards[board].name);
	dev->board_name=ni_boards[board].name;
	dev->driver_name="pcimio-E";

	dev->private=kmalloc(sizeof(ni_private),GFP_KERNEL);
	if(!dev->private){
		printk(" no memory\n");
		return -ENOMEM;
	}
	memset(dev->private,0,sizeof(ni_private));
	
	devpriv->board=board;
	
	setup_mite(dev);
	
        /* irq stuff */

        irq=devpriv->pci_device->irq;
        if(irq==0){
		printk(" unknown irq (bad)\n");
		return -EINVAL;
	}
        printk(" ( irq = %d )",irq);
        if( (ret=request_irq(irq,ni_E_interrupt,SA_INTERRUPT,"pcimio-E",dev))<0 ){
                printk(" irq not available\n");
                return -EINVAL;
        }
        dev->irq=irq;

	if( (ret=ni_E_init(dev,it))<0 ){
		
		return ret;
	}
	
	return 0;
}


/*
 *  this function is similar to pci_find_device in
 *  linux/drivers/pci/pci.c
 *
 */

static struct pci_dev *
ni_pci_find_device(struct pci_dev *from,int *board)
{
	int i;
	
	if(!from){
		from = pci_devices;
	}else{
		from = from->next;
	}
	while (from){
		if( from->vendor == PCI_VENDOR_ID_NATINST){
			for(i=0;i<n_ni_boards;i++){
				if(from->device == ni_boards[i].device_id){
					*board=i;
					return from;
				}
			}
			printk("unknown NI device found with device_id=0x%04x\n",from->device);
		}
		from = from->next;
	}
	*board=-1;
	return from;
}

static int setup_mite(comedi_device *dev)
{
	unsigned long			offset;
	u32				addr;

	addr=devpriv->pci_device->base_address[1];
	devpriv->daq_phys_addr=addr;
	offset = devpriv->daq_phys_addr & ~PAGE_MASK;
	devpriv->daq_io_addr = ioremap(devpriv->daq_phys_addr & PAGE_MASK, PCI_DAQ_SIZE + offset )
		+ offset;
#ifdef PCI_DEBUG
	printk("DAQ:0x%08lx mapped to %p, ",devpriv->daq_phys_addr,devpriv->daq_io_addr);
#endif

	addr=devpriv->pci_device->base_address[0];
	devpriv->mite_phys_addr=addr;
	offset = devpriv->mite_phys_addr & ~PAGE_MASK;
	devpriv->mite_io_addr = ioremap(devpriv->mite_phys_addr & PAGE_MASK, PCI_MITE_SIZE + offset )
		+ offset;
#ifdef PCI_DEBUG
	printk("MITE:0x%08lx mapped to %p ",devpriv->mite_phys_addr,devpriv->mite_io_addr);
#endif
	
	/* XXX don't know what the 0xc0 and 0x80 mean */
	writel(devpriv->daq_phys_addr | 0x80 , devpriv->mite_io_addr + 0xc0 );
	
	dev->iobase = (int) devpriv->daq_io_addr;
	
	return 0;
}

#endif


static void unset_mite(comedi_device *dev)
{
	if(devpriv->mite_io_addr){
		iounmap(devpriv->mite_io_addr);
		devpriv->mite_io_addr=NULL;
	}
	if(devpriv->daq_io_addr){
		iounmap(devpriv->daq_io_addr);
		devpriv->daq_io_addr=NULL;
	}
}


