/*
    module/timer.c
    polling routines

    COMEDI - Linux Control and Measurement Device Interface
    Copyright (C) 1997-8 David A. Schleef <ds@stm.lbl.gov>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/fcntl.h>
#include <linux/timer.h>
#include <comedi_module.h>

/*
 *  polling routines
 *
 *  how to use:
 *
 *  - at initialization, hardware drivers assign dev->poll.
 *    this is the function that is called at polling time 
 *
 *  - hardware drivers call start_polling() when they need polling.
 *
 *  - polling routines or whatever calls end_polling(dev) when no
 *    part of the device needs polling
 *
 *  how it works:
 *  
 *  - a kernel timer is started when start_polling() is called and
 *    the timer is not already running
 *
 *  - the polling routine is called when the timer expires, which
 *    calls hardware driver polling routines.  it searches the devices
 *    to determine if it should continue polling, and if so,
 *    resets the timer.
 */

#undef DEBUG_POLLING

static struct timer_list comedi_timer;
static int poll_running=0;

static void comedi_poll(unsigned long d);


void init_polling(void)
{
	init_timer(&comedi_timer);
	comedi_timer.data=0;
	comedi_timer.function=comedi_poll;
}

void cleanup_polling(void)
{
	if(poll_running){
		printk("comedi: poll running at module cleanup!\n");
		del_timer(&comedi_timer);
	}
}

void start_polling(comedi_device *dev)
{
#ifdef DEBUG_POLLING
	printk("comedi: start_polling\n");
#endif
	dev->polling=1;
	if(!poll_running){
		poll_running=1;
		comedi_timer.expires=jiffies+1;
		add_timer(&comedi_timer);
	}
}

void stop_polling(comedi_device *dev)
{
#ifdef DEBUG_POLLING
	printk("comedi: stop_polling\n");
#endif
	dev->polling=0;
}

static void comedi_poll(unsigned long d)
{
	int i;
	int again=0;
	
	for(i=0;i<COMEDI_NDEVICES;i++){
		if(comedi_devices[i].polling && comedi_devices[i].poll)
			comedi_devices[i].poll(comedi_devices+i);
		again|=comedi_devices[i].polling;
	}
	if(comedi_timer.expires<jiffies){
		printk("comedi: lost a jiffy\n");
	}
	if(again){
		comedi_timer.expires=jiffies+1;
		add_timer(&comedi_timer);
	}else{
		poll_running=0;
	}
}

