/*
    module/comedi_module.h
    header file for kernel-only structures, variables, and constants

    COMEDI - Linux Control and Measurement Device Interface
    Copyright (C) 1997-8 David A. Schleef <ds@stm.lbl.gov>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef _COMEDI_MODULE_H
#define _COMEDI_MODULE_H

#include <linux/version.h>
#include <linux/config.h>
#include <linux/kdev_t.h>
#include <config.h>
#include <linux/malloc.h>
#include <linux/errno.h>

#if LINUX_VERSION_CODE < 0x020000
#error kernel versions prior to 2.0 not supported
#else
#if LINUX_VERSION_CODE < 0x020100
#define LINUX_V20
#else
#define LINUX_V22
#endif
#endif

#include <comedi.h>

#ifdef CONFIG_DEBUG
#define DPRINTK(format, args...)	printk( format , ## args )
#else
#define DPRINTK(format, args...)	/* */
#endif

typedef struct comedi_device_struct comedi_device;
typedef struct comedi_subdevice_struct comedi_subdevice;

typedef int comedi_devinit(comedi_device *,comedi_devconfig *);

typedef struct comedi_bufll{
	struct comedi_bufll *next;
	int use_count;
	char stuff[0];
}comedi_bufll;

typedef struct{
	comedi_bufll *first;
	comedi_bufll *last;
	int in;
	int use_count;
	int num_pages;
}comedi_buf;

typedef struct{
	comedi_buf *buf;
	comedi_bufll *bufll;
	int offset;
}comedi_bufptr;

struct comedi_subdevice_struct{
	int type;
	int n_chan;
	int subdev_flags;
	int timer_type;
	int len_chanlist;		/* length of channel/gain list, if available */
	unsigned int maxbufsz;

	void *lock;
	void *busy;
	
	int io_bits;
	
	sampl_t maxdata;		/* if maxdata==0, use list */
	sampl_t *maxdata_list;		/* list is channel specific */
	
	unsigned int flags;
	unsigned int *flaglist;

	unsigned int range_type;
	unsigned int *range_type_list;
	
	unsigned int *chanlist;		/* driver-owned chanlist (not used) */
	
	comedi_trig	cur_trig;	/* current trig structure */
	
	sampl_t		*buf;		/* pre-allocated buffer */

	unsigned int buf_len;
	volatile unsigned int buf_int_ptr;	/* buffer marker for interrupt */
	unsigned int buf_user_ptr;	/* buffer marker for read() and write() */
	unsigned int cur_chan;		/* useless channel marker for interrupt */
	
	unsigned int *range_list;	/* is this necessary? */
	
	int (*trig[5])(comedi_device *,comedi_subdevice *,comedi_trig *);

	int (*cancel)(comedi_device *,comedi_subdevice *);

	int (*end_of_scan)(void *);
	int (*end_of_acquisition)(void *);

	void *eos_arg;
	void *eoa_arg;

	unsigned int state;
};

struct comedi_device_struct{
	int use_count;
	void *private;
	kdev_t minor;
	int (*rem)(comedi_device *);
	char *driver_name;
	char *board_name;
	int board;			/* this would be more useful as void* */

	int n_subdevices;
	comedi_subdevice *subdevices;
	int options[COMEDI_NDEVCONFOPTS];

	/* dumb */
	int iobase;
	int iosize;
	int irq;

	struct wait_queue *wait;

	/* still needed? */
	comedi_buf rdbuf;
	void (*poll)(comedi_device *);
	int polling;

};



extern comedi_device *comedi_devices;
extern comedi_devinit *comedi_devinits[];
extern const int comedi_maxinits;

/*
 * function prototypes
 */

void comedi_error(comedi_device *dev,const char *s);
void comedi_done(comedi_device *dev,comedi_subdevice *s);
void comedi_eos(comedi_device *dev,comedi_subdevice *s);

void init_polling(void);
void cleanup_polling(void);
void start_polling(comedi_device *);
void stop_polling(comedi_device *);

void comedi_proc_init(void);
void comedi_proc_cleanup(void);

/*
   various internal comedi functions
 */

int do_rangeinfo_ioctl(comedi_device *dev,comedi_rangeinfo *arg);
int check_chanlist(comedi_subdevice *s,int n,unsigned int *chanlist);

/* range stuff */

#include <range.h>

#define RANGE_digital			RANGE_unipolar5

extern unsigned int comedi_max_range;
extern comedi_krange comedi_kranges[];


/* timer types */
/* these *must* agree with lib/timer.c */

#define TIMER_dt282x			1
#define TIMER_dt2814			2
#define TIMER_atmio			3
#define TIMER_acl8112			4


/* some silly little inline functions */

static inline int alloc_subdevices(comedi_device *dev)
{
	int size=sizeof(comedi_subdevice)*dev->n_subdevices;

	dev->subdevices=kmalloc(size,GFP_KERNEL);
	if(!dev->subdevices)
		return -ENOMEM;
	memset(dev->subdevices,0,size);
	return 0;
}

static inline int alloc_private(comedi_device *dev,int size)
{
	dev->private=kmalloc(size,GFP_KERNEL);
	if(!dev->private)
		return -ENOMEM;
	memset(dev->private,0,size);
	return 0;
}

/* kernel compatibility */

#ifdef LINUX_V22

/* this is a useless abstraction that keeps several #ifdefs out of module.c */
#define minor_of_file(x)        MINOR((x)->f_dentry->d_inode->i_rdev)

#endif

#ifdef LINUX_V20

#define minor_of_file(x)        MINOR((x)->f_inode->i_rdev)

#define ioremap vremap
#define iounmap vfree

#define signal_pending(x)	(((x)->signal) & (~(x)->blocked))

#if CONFIG_EXPORT
extern struct symbol_table comedi_syms;
#endif

#include <asm/segment.h>

static inline int copy_to_user(void * to,const void *from,unsigned long n_bytes)
{
	int i;

	if((i=verify_area(VERIFY_WRITE,to,n_bytes)) != 0)
		return i;
	memcpy_tofs(to,from,n_bytes);
	return 0;
}

static inline int copy_from_user(void * to,const void *from,unsigned long n_bytes)
{
	int i;
	if((i=verify_area(VERIFY_READ,from,n_bytes))!=0)
		return i;
	memcpy_fromfs(to,from,n_bytes);
	return 0;
}

static inline int clear_user(void * mem,unsigned long len)
{
	char *cmem=mem;
	
	if(verify_area(VERIFY_WRITE,mem,len))
		return len;
	/* this is slow, but I'm lazy */
	while(len--){put_user(0,cmem);cmem++;}
	return 0;
}

#endif


#ifdef LINUX_V20
/* actually LINUX_VERSION_CODE < 0x020155 */

#define LINUX_old_PCI_compatibility

#endif


#endif /* _COMEDI_MODULE_H */




