/*
 * Device Driver for DataTranslation DT2801
 *
 */

#include <comedi_module.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <asm/io.h>

/* c.f. Linux Journal 3/96 p.13 */
#ifdef DEBUG_DT2801
#define DT_PRINTK(format, args...)  printk ("dt2801:" format, ## args)
#define DT_PRINTKF(format, args...)  printk ("dt2801:" "%s: " format, __FUNCTION__, ## args)
#else
#define DT_PRINTK(format, args...)  
#define DT_PRINTKF(format, args...) 
#endif


/* Hardware Configuration */
/* ====================== */

/*
 *  General
 */
#define DT2801_MAX_TYPES 8

#define DT2801_T_2801		(0x09)
#define DT2801_T_2801A		(0x52)
#define DT2801_T_2801_5716A	(0x82)
#define DT2801_T_2805		(0x12)
#define DT2801_T_2805_5716A	(0x92)
#define DT2801_T_2808		(0x20)
#define DT2801_T_2818		(0xA2)
#define DT2801_T_2809		(0xB0)


#define DT2801_MAX_DMA_SIZE (64 * 1024)

/* Ports */
#define DT2801_IOSIZE 2


/* define's & typedef's */
/* ====================== */

/* Commands */
#define DT_C_RESET       0x0
#define DT_C_CLEAR_ERR   0x1
#define DT_C_READ_ERRREG 0x2
#define DT_C_SET_CLOCK   0x3

#define DT_C_TEST        0xb
#define DT_C_STOP        0xf

#define DT_C_SET_DIGIN   0x4
#define DT_C_SET_DIGOUT  0x5
#define DT_C_READ_DIG    0x6
#define DT_C_WRITE_DIG   0x7

#define DT_C_WRITE_DAIM  0x8
#define DT_C_SET_DA      0x9
#define DT_C_WRITE_DA    0xa

#define DT_C_READ_ADIM   0xc
#define DT_C_SET_AD      0xd
#define DT_C_READ_AD     0xe

/* Command modifiers (only used with read/write), EXTTRIG can be
   used with some other commands.
*/
#define DT_MOD_DMA     (1<<4)
#define DT_MOD_CONT    (1<<5)
#define DT_MOD_EXTCLK  (1<<6)
#define DT_MOD_EXTTRIG (1<<7)

/* Bits in status register */
#define DT_S_DATA_OUT_READY   (1<<0)
#define DT_S_DATA_IN_FULL     (1<<1)
#define DT_S_READY            (1<<2)
#define DT_S_COMMAND          (1<<3)
#define DT_S_COMPOSITE_ERROR  (1<<7)

#define DT2801_TIMERCYCLES 2

/* registers */
#define DT2801_DATA		0
#define DT2801_STATUS		1
#define DT2801_CMD		1


typedef struct{
	char *name;
	int boardcode;
	int adbits;
	int adrangetype;
} boardtype_t;

/* Typeid's for the different boards of the DT2801-series
   (taken from the test-software, that comes with the board)
   */
static boardtype_t boardtypes[] =
{
	{"dt2801",0x09,		12,RANGE_unknown},
	{"dt2801-a",0x52,	12,RANGE_unknown},
	{"dt2801/5716a",0x82,	16,RANGE_unknown},
	{"dt2805",0x12,		12,RANGE_unknown},
	{"dt2805/5716a",0x92,	16,RANGE_unknown},
	{"dt2808",0x20,		12,RANGE_unknown},
	{"dt2818",0xa2,		12,RANGE_unknown},
	{"dt2809",0xb0,		12,RANGE_unknown},
};


typedef struct{
	boardtype_t *board;
}dt2801_private;
#define devpriv ((dt2801_private *)dev->private)
#define boardtype (*devpriv->board)


static int dt2801_ai_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);
static int dt2801_ao_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);
static int dt2801_dio_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);

/* These are the low-level routines:
   writecommand: write a command to the board
   writedata: write data byte
   readdata: read data byte

   They check the status. If the board is not ready the sleep by
   calling dt2801_wait() and try again. If the board is still not
   ready they return with a value != 0 */


void dt2801_wait(comedi_device *dev)
{
	udelay(100);
#if 0
	dt2801_start_timer(boardPtr);
 
	interruptible_sleep_on(&boardPtr->waitq);
	boardPtr->dt_timer_int = 0;
	del_timer(&boardPtr->dt_timer_list);
#endif
}


/* Only checks DataOutReady-flag, not the Ready-flag as it is done
   in the examples of the manual. I don't see why this should be
   necessary. */
static int dt2801_readdata(comedi_device *dev, int* data)
{
	int stat = 0;

	stat = inb_p(dev->iobase+DT2801_STATUS);
	if (stat & DT_S_COMPOSITE_ERROR) {
            return -1;
	}

	if (!(stat&DT_S_DATA_OUT_READY)) {
		dt2801_wait(dev);

		stat = inb_p(dev->iobase+DT2801_STATUS);

		if (!(stat&DT_S_DATA_OUT_READY)) {
			printk("dt2801: readdata timed out. "
			       "DataOutReady not set\n");
			return -1;
		}

	}
	*data = inb_p(dev->iobase+DT2801_DATA);

	return 0;
}

static int dt2801_readdata2(comedi_device *dev, int *data)
{
	int lb, hb;

	if (dt2801_readdata(dev, &lb) != 0) {
		return -1;
	}
	if (dt2801_readdata(dev, &hb) != 0) {
		return -2;
	}

	*data =  (hb<<8)+lb;
	return 0;
}

static int dt2801_writedata(comedi_device *dev, unsigned int data)
{
	int stat = 0;

	stat = inb_p(dev->iobase+DT2801_STATUS);

	if (stat & DT_S_COMPOSITE_ERROR) {
            return -1;
	}
	if (stat & DT_S_DATA_IN_FULL) {
		dt2801_wait(dev);
		stat = inb_p(dev->iobase+DT2801_STATUS);

		if (stat & DT_S_DATA_IN_FULL) {
			printk("dt2801: writedata timed out. "
				"DataInFull not cleared\n");
			return -1;
		}
	}
	outb_p(data & 0xff, dev->iobase+DT2801_DATA);

	return 0;
}

static int dt2801_writedata2(comedi_device *dev, unsigned int data)
{

	if ( dt2801_writedata(dev,  data & 0xff) != 0) {
		return -1;
	}
	if ( dt2801_writedata(dev,  (data >> 8) & 0xff ) != 0) {
		return -1;
	}

	return 0;
}

static int dt2801_writecmd(comedi_device * dev, int command)
{
	int stat;

	stat = inb_p(dev->iobase+DT2801_STATUS);
	if (stat & DT_S_COMPOSITE_ERROR) {
        	printk("dt2801: writecommand : warning composite-error\n");
	}

	if (stat & DT_S_DATA_IN_FULL) {
		dt2801_wait(dev);

		stat = inb_p(dev->iobase+DT2801_STATUS);

		if (stat & DT_S_DATA_IN_FULL) {
			printk("dt2801: writecommand timed out. "
			       "DataInFull not cleared\n");
			return -1;
		}
	}

	stat = inb_p(dev->iobase+DT2801_STATUS);

	if (!(stat&DT_S_READY)) {
		dt2801_wait(dev);

		stat = inb_p(dev->iobase+DT2801_STATUS);

		if (!(stat&DT_S_READY)) {
			printk("dt2801: writecommand timed out."
			       "Ready not set\n");
			return -1;
		}

	}
	outb_p(command, dev->iobase+DT2801_CMD);
	return 0;
}

static boardtype_t * dt2801_get_boardtype(comedi_device *dev)
{
	return boardtypes;

}

static int dt2801_do_reset(comedi_device *dev)
{
	int stat, board_type;

	stat = inb_p(dev->iobase+DT2801_STATUS);
	/* printk("dt2801: before reset: status = 0x%0x\n", stat); */
	/* stat != 0 if DataInFull-Flag is cleared */
	stat= ((stat)^DT_S_DATA_IN_FULL) & DT_S_DATA_IN_FULL;
        if (!stat) {
		dt2801_wait(dev);
		stat = inb_p(dev->iobase+DT2801_STATUS);
		stat= ((stat)^DT_S_DATA_IN_FULL) & DT_S_DATA_IN_FULL;
		if (!stat) {
			printk("dt2801: do_reset timed out. "
			       "DataInFull not cleared\n");
			printk("dt2801: status = 0x%0x\n", stat);
		        return -1;
		}
	}

	stat = inb_p(dev->iobase+DT2801_STATUS);
	/* printk("dt2801: rst-1 status = 0x%0x\n", stat); */
	/* stat != 0 if Ready-Flag is set */
	stat = stat & DT_S_READY;
        if (!stat) {
		dt2801_wait(dev);
		stat = inb_p(dev->iobase+DT2801_STATUS);
		stat = stat & DT_S_READY;
		if (!stat) {
			printk("dt2801: do_reset timed out. "
			       "Ready not set\n");
			printk("dt2801: status = 0x%0x\n", stat);
		        return -1;
		}
	}
	outb_p(DT_C_RESET, dev->iobase+DT2801_CMD);

	/* This is basically readdata(), but completes if
	   Ready OR DataOutReady are set. */
	stat = inb_p(dev->iobase+DT2801_STATUS);
	/* printk("dt2801: rst-2 status = 0x%0x\n", stat); */
	stat = stat & (DT_S_DATA_OUT_READY);
        if (!stat) {
		dt2801_wait(dev);
		stat = inb_p(dev->iobase+DT2801_STATUS);
	        /* printk("dt2801: rst-3 status = 0x%0x\n", stat); */
		stat = stat & ( DT_S_DATA_OUT_READY);
		if (!stat) {
			printk("dt2801: do_reset timed out. "
			       "Ready not set\n");
			printk("dt2801: status = 0x%0x\n", stat);
		        return -1;
		}
	}
	dt2801_wait(dev);
	board_type = inb_p(dev->iobase+DT2801_DATA);
	dt2801_wait(dev);

	stat = inb_p(dev->iobase+DT2801_STATUS);
	/* printk("dt2801: after reset: status = 0x%0x\n", stat); */

	return board_type;
}

static int dt2801_rem(comedi_device *dev);
/*
   options:
	[0] - i/o base
	[1] - unused
	[x] - a/d 0=differential, 1=single-ended
	[x] - dac0 unipolar=0, bipolar=1
	[x] - dac0 5 V reference =0, 10 V ref = 1
	[x] - dac1 unipolar=0, bipolar=1
	[x] - dac0 5 V reference =0, 10 V ref = 1
*/

int dt2801_init(comedi_device *dev,comedi_devconfig *it)
{
	comedi_subdevice *s;
	int iobase;
	int ret=0;

	if(strcmp("dt2801",it->board_name))
		return 0;

	iobase=it->options[0];
	if(check_region(iobase,DT2801_IOSIZE)<0){
		comedi_error(dev,"I/O port conflict");
		return -EIO;
	}
	request_region(dev->iobase, DT2801_IOSIZE, "dt2801");
	dev->iobase=iobase;

	/* do some checking */

	dt2801_do_reset(dev);

	devpriv->board = dt2801_get_boardtype(dev);

	dev->n_subdevices=4;

	if((ret=alloc_subdevices(dev))<0)
		goto cleanup;

	if((ret=alloc_private(dev,sizeof(dt2801_private)))<0)
		goto cleanup;

	s=dev->subdevices+0;
	/* ai subdevice */
	s->type=COMEDI_SUBD_AI;
	s->subdev_flags=SDF_READABLE;
	s->n_chan=8;
	s->maxdata=(1<<boardtype.adbits)-1;
	s->timer_type=0;	/* XXX fix */
	s->range_type=boardtype.adrangetype;
	s->trig[0]=dt2801_ai_mode0;

	s++;
	/* ao subdevice */
	s->type=COMEDI_SUBD_AO;
	s->subdev_flags=SDF_WRITEABLE;
	s->n_chan=2;
	s->maxdata=0xfff;
	s->range_type=RANGE_unknown;
	s->trig[0]=dt2801_ao_mode0;

	s++;
	/* 1st digital subdevice */
	s->type=COMEDI_SUBD_DIO;
	s->subdev_flags=SDF_READABLE|SDF_WRITEABLE;
	s->n_chan=8;
	s->maxdata=1;
	s->range_type=RANGE_digital;
	s->trig[0]=dt2801_dio_mode0;

	s++;
	/* 2nd digital subdevice */
	s->type=COMEDI_SUBD_DIO;
	s->subdev_flags=SDF_READABLE|SDF_WRITEABLE;
	s->n_chan=8;
	s->maxdata=1;
	s->range_type=RANGE_digital;
	s->trig[0]=dt2801_dio_mode0;

	dev->rem=dt2801_rem;

	return 1;
cleanup:
	dt2801_rem(dev);

	return ret;
}

static int dt2801_rem(comedi_device *dev)
{
	if(dev->subdevices)
		kfree(dev->subdevices);

	if(dev->private)
		kfree(dev->private);

	if(dev->iobase)
		release_region(dev->iobase,DT2801_IOSIZE);

	return 0;
}

static int dt2801_ai_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
        int data;
        int stat;


        stat = dt2801_writecmd(dev, DT_C_READ_ADIM);
        dt2801_writedata(dev, CR_RANGE(it->chanlist[0]));
        dt2801_writedata(dev, CR_CHAN(it->chanlist[0]));
        stat = dt2801_readdata2(dev, &data);

        if (stat != 0) {
             printk("dt2801: stat = %x\n", stat);
             return -EIO;
        }

	it->data[0]=data;

        return 1;
}

static int dt2801_ao_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	int chan=CR_CHAN(it->chanlist[0]);

	dt2801_writecmd(dev, DT_C_WRITE_DAIM);
	dt2801_writedata(dev, chan);
	dt2801_writedata2(dev, it->data[0]);
	
	return 1;
}

static int dt2801_dio_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	return -EINVAL;
}




/*
   end of comedi-safe stuff
 */





#if 0
/* Parameters for the whole board.
   Some are fixed through jumper settings. But I put these into
   the driver so they can be reported to programs. For now these
   will be given at module load-time. May make them changeable at
   runtime in the future. */

typedef struct Dt2801_t Dt2801_t;


typedef struct {
	/* The clock (ticks) */
	int min;
	int max;
	int ticks;
} Clock_state;

struct Dt2801_t {

	int boardid;
	int typeid;
	char* typename;

	int fake;
	struct timer_list tl;
	int fake_counter;
	int stop_req;
	int running;

	unsigned long baseaddr;
	int dmachannel;

	int inuse;
	int usecount;

	int inerror;
	int firstopen;
	int status;
	int dma_running;
	int exttrigger_expected;

        Clock_state clock;

        ADC_state adc_state;
        DAC_state dac_state;
        DIO_state dio_state;

	int dt_timer_int;
	struct timer_list dt_timer_list;
	struct wait_queue* waitq;
};
#endif




#if 0
extern int dt2801_do_ioctl(struct inode*, struct file*,
			   unsigned int, unsigned long, Dt2801_t*);
extern void dt2801_release(struct inode*, struct file*);
extern int dt2801_open(struct inode*, struct file*);
extern int dt2801_stop(Dt2801_t*);
extern int dt2801_do_stop(Dt2801_t*);
extern int dt2801_clearerror(Dt2801_t*);
extern int dt2801_reset(Dt2801_t*);
extern int dt2801_checkforerror(Dt2801_t*);
#endif


/* A/D-converter */

#define DT2801_NUM_ADCS 1
#define DT2801_ADC_CHANNELS 8

/* Minor device numbers */
#define DT2801_ADCX     0x1f /* the generic adc-device */


#define DT2801_ADC_UNIPOLAR 0
#define DT2801_ADC_BIPOLAR  1

#define DT2801_ADC_DIFFERENTIAL 0
#define DT2801_ADC_SINGLE_ENDED 1


#if 0
extern struct file_operations dt2801_adc_fops;
extern int dt2801_adc_init(Dt2801_t*);
extern void dt2801_adc_info(Dt2801_t*);
extern int dt2801_adc_cleanup(Dt2801_t* boardPtr);
extern int dt2801_adc_reset(Dt2801_t* boardPtr);
extern int dt2801_adc_open(struct inode *inode, struct file *filp);
#endif

#if 0
typedef struct {
        int bits;
	int polarity;
	int inputmode;
	int nchannels;

        /* 'gain', 'startch', 'endch' as set by the last
	   set_adc_params command */
        int gain;

        int startch;
        int endch;
        int conv;

} ADC_state;

typedef struct {
        Dt2801_t* board;

	int exttrigger;
	int extclock;
	int continous;

        int startch;
        int endch;
        int gain;
        int conv;

        /* XXX  for DMA-transfers*/
	int last;
	int currnt;
	int running;
	int cnt;

	int dmabufsize;
	unsigned short* dmabuf;
	int sz;
} ADCP;


typedef struct {
    int startch;
    int endch;
    int gain;
    int conv;
} Dt2801_ADCP_t;
#endif


#define DT2801_IOCTL_ADC_SET_PARAMS _IOW(DT2801_MAGIC, 0x11, Dt2801_ADCP_t)
#define DT2801_IOCTL_ADC_BLOCKSTART _IO(DT2801_MAGIC, 0x12)
#define DT2801_IOCTL_ADC_BLOCKSTOP  _IO(DT2801_MAGIC, 0x13)
#define DT2801_IOCTL_ADC_RESIDUE    _IOR(DT2801_MAGIC, 0x14, int)
#define DT2801_IOCTL_ADC_SETEXTCLOCK _IOW(DT2801_MAGIC, 0x15, int)
#define DT2801_IOCTL_ADC_SETCONTINOUS _IOW(DT2801_MAGIC, 0x16, int)
#define DT2801_IOCTL_ADC_GET_POLARITY   _IOR(DT2801_MAGIC, 0x17, int)
#define DT2801_IOCTL_ADC_SET_GAIN       _IOW(DT2801_MAGIC, 0x18, int)
#define DT2801_IOCTL_ADC_SETEXTTRIGGER _IOW(DT2801_MAGIC, 0x19, int)


/*
 * Device Driver for DataTranslation DT2801
 *
 * Digital-Analog-Converter subsystem
 *
 */

/* There are 2 DAC-channels (12-bit reslution), which can be
   configured separately. Either unipolar output (0..5V /
   0..10V full-scale), or bipolar output (-2.5..+2.5V / -5.0..+5.0 /
   -10.0..+10.0).
   This is done with jumpers, so not configurable at runtime. */

#define DT2801_DAC_UNIPOLAR 0
#define DT2801_DAC_BIPOLAR  1


/* All Dt2801-series boards have two DACs */
#define DT2801_NDACS 2

#if 0
typedef struct {
        int bits;
        int fullscale;
	int polarity;
	int usecount;
} DAC_t;

typedef struct {
	int conv;
	int dacnr;
        DAC_t dacs[DT2801_NDACS];
} DAC_state;

typedef struct {
        Dt2801_t* board;

	int mode;
	int dacnr;
	int exttrigger;
} DACP;
#endif



/* Minor device numbers */
#define DT2801_DAC0     0x00
#define DT2801_DAC1     0x01
#define DT2801_DAC2     0x02

extern struct file_operations dt2801_dac_fops;
#if 0
extern int dt2801_dac_init(Dt2801_t*);
extern void dt2801_dac_info(Dt2801_t*);
extern int dt2801_dac_cleanup(Dt2801_t*);
extern int dt2801_dac_reset(Dt2801_t*);
extern int dt2801_dac_open(struct inode *inode, struct file *filp);
#endif


typedef struct {
	unsigned int conv;
} Dt2801_DACP_t;


#define DT2801_DAC_IMMEDIATE 1
#define DT2801_DAC_BLOCK     2

#define DT2801_IOCTL_DAC_SET_PARAMS _IOW(DT2801_MAGIC, 0x21, Dt2801_DACP_t*)
#define DT2801_IOCTL_DAC_GET_POLARITY  _IOR(DT2801_MAGIC, 0x22, int)
#define DT2801_IOCTL_DAC_GET_FULLSCALE  _IOR(DT2801_MAGIC, 0x23, int)
#define DT2801_IOCTL_DAC_SET_MODE  _IOW(DT2801_MAGIC, 0x24, int)

/*
 * Device Driver for DataTranslation DT2801
 *
 *
 */


#define DT2801_DIO_NPORTS 2 
#define DT2801_DIO_PORTSIZE 8 

/* Minor device numbers */
#define DT2801_DIO0     0x30  /* port 0, 8-bit */
#define DT2801_DIO1     0x31  /* port 1, 8-bit */
#define DT2801_DIO2     0x32  /* port 0 and 1 treated as one 16-bit port */
#define DT2801_DIOX     0x3f  /* the generic dio-device */
#define DT2801_DIOL0    0x20  /* single-line device */
#define DT2801_DIOL15   0x2f  /* single-line device */

#define DT2801_DIO_P0     (0)
#define DT2801_DIO_P1     (1)
#define DT2801_DIO_P2     (2)


#define DT2801_DIO_PORTUNDEF -1
#define DT2801_DIO_PORTIN 0 
#define DT2801_DIO_PORTOUT 1 

typedef unsigned short uint16;

#if 0
typedef struct {
	int usecount;
	int mode;
	int lines; /* lower 8 bits only */
} DIO_port;

typedef struct {
	int usecount;
	int nports;
        DIO_port ports[DT2801_DIO_NPORTS];
} DIO_state;

typedef struct {
        Dt2801_t* board;

	uint16 lines;
	int port;
	int mode;
	int exttrigger;
} DIOP;
#endif

#if 0
extern struct file_operations dt2801_dio_fops;
extern int dt2801_dio_init(Dt2801_t*);
extern void dt2801_dio_info(Dt2801_t*);
extern int dt2801_dio_cleanup(Dt2801_t*);
extern int dt2801_dio_reset(Dt2801_t*);
extern int dt2801_dio_open(struct inode *inode, struct file *filp);
#endif


#define DT2801_IOCTL_DIO_SET_LINES _IOW(DT2801_MAGIC, 0x31, uint16)

/*
 * Device Driver for DataTranslation DT2801
 *
 * Analog-Digital-Converter subsystem
 *
 */


/* routines for faked driver operation, allows to test the driver
   when no hardware is present ( fake=1 as argument, to activate) */


#if 0
static void adc_start_timer(ADCP* adcp);

static void adc_timer_interrupt(unsigned long bp)
{
	ADCP* adcp = (ADCP*) bp;
	Dt2801_t* boardPtr = adcp->board;

        del_timer(&boardPtr->tl);
        if (boardPtr->fake != 0) {
                short *b = (short *) adcp->dmabuf;

                b[boardPtr->fake_counter] = boardPtr->fake_counter%5;
                boardPtr->fake_counter++;
		adc_start_timer(adcp);
        }
}

static void adc_start_timer(ADCP* adcp)
{
	Dt2801_t* boardPtr = adcp->board;

        if (boardPtr->fake_counter >= adcp->conv) {
		boardPtr->running = 0;
                return;
        }
	if (boardPtr->stop_req != 0) {
		boardPtr->running = 0;
                return;
	}

        init_timer(&boardPtr->tl);
        boardPtr->tl.expires = jiffies + 20;
	boardPtr->tl.data = (unsigned long) adcp;
        boardPtr->tl.function = adc_timer_interrupt;
        add_timer(&boardPtr->tl);
}

/* end of fake routines */

static int allocate_mmapadc_buffer(ADCP* adcp, int bufsize)
{
	int sz, size, i;
	char* start_addr, *end_addr;

        if (bufsize <= 0) {
                printk("dt2801_adc: allocating DMA-buffer,"
                       " illegal bufsize %d\n", bufsize);
                return -(EINVAL);
        }
        if (bufsize > DT2801_MAX_DMA_SIZE) {
                printk("dt2801_adc: Requested DMA buffer size,"
                       " %d, too large\n", bufsize );
                return -(EINVAL);
        }

        /* reuse the buffer if it is large enough */
        if (bufsize <= adcp->dmabufsize ) {
                return 0;
        }
 
        if (adcp->dmabuf != NULL) {
                printk("dt2801_adc: freeing dmabuf\n");
                kfree(adcp->dmabuf);
                adcp->dmabufsize = 0;
                adcp->dmabuf = NULL;
        }

        /* Allocate buffer for DMA transfers */

	/* mmapable */
	/* Taken from the sounddriver. It's not clear to
	   me what the requirements for these pages are
	   to make the mmap work, but this one does. */

	for (sz = 0, size = PAGE_SIZE; size < bufsize; sz++, size <<= 1);

	printk("dt2801: sz = %d\n", sz);

        adcp->dmabufsize = PAGE_SIZE * (1 << sz);
	start_addr =  __get_free_pages(GFP_KERNEL, sz, MAX_DMA_ADDRESS);
	if (start_addr == NULL) {
		printk("dt2801: __get_free_pages failed\n");
		return  -(ENOMEM);
	}
	end_addr = start_addr + adcp->dmabufsize - 1;
	for (i = MAP_NR(start_addr); i <= MAP_NR(end_addr); i++) {
		mem_map_reserve (i);
	}
	adcp->dmabuf = start_addr;
	adcp->sz     = sz;

        return 0;
}

static int allocate_adc_buffer(ADCP* adcp, int bufsize)
{
        if (bufsize <= 0) {
                printk("dt2801_adc: allocating DMA-buffer,"
                       " illegal bufsize %d\n", bufsize);
                return -(EINVAL);
        }
        if (bufsize > DT2801_MAX_DMA_SIZE) {
                printk("dt2801_adc: Requested DMA buffer size,"
                       " %d, too large\n", bufsize );
                return -(EINVAL);
        }

        /* reuse the buffer if it is large enough */
        if (bufsize <= adcp->dmabufsize ) {
                return 0;
        }
 
        if (adcp->dmabuf != NULL) {
                printk("dt2801_adc: freeing dmabuf\n");
                kfree(adcp->dmabuf);
                adcp->dmabufsize = 0;
                adcp->dmabuf = NULL;
        }

        /* Allocate buffer for DMA transfers */
        adcp->dmabuf = kmalloc( bufsize, GFP_KERNEL | GFP_DMA );
        if (adcp->dmabuf == NULL) {
                printk( "dt2801_adc: Unable to allocate DMA buffer\n" );
                return -(ENOMEM);
        }
        adcp->dmabufsize = bufsize;

        return 0;
}

static void clear_adc_buffer(ADCP* adcp)
{
        int i;

        for (i=0; i<adcp->dmabufsize; i++) {
                adcp->dmabuf[i] = 0xf000;
        }
}

/* Find the first 2 byte value in the buffer, that is not yet
   complete. Only works with 12 bit ADC, not with 16 bit*/
static int get_currnt(ADCP* adcp)
{
        int i;

	/* This tests whether the highbyte still has its high nibble
	   set to 0xf, as it was done by 'clear_adc_buffer' This
	   works only on a little endian machine. This is necessary
	   to see, if the 2 bytes (12 bit significant) of the
	   conversion have both arrived. */
	for (i=adcp->cnt; i<adcp->conv; i++) {
		if (adcp->dmabuf[i] & 0xf000) {
                        break;
                }
        }

        if (i==0) {
		adcp->cnt = i;
	} else if (i == adcp->conv) {
		adcp->cnt = i;
	} else {
		adcp->cnt = i-1;
	}
        return adcp->cnt;
}

static int dt2801_set_adc_params(ADCP* adcp, Dt2801_ADCP_t *adcpar)
{                
        int stat;
	Dt2801_t* boardPtr = adcp->board;

        dt2801_writecommand(boardPtr, DT_C_SET_AD);

        dt2801_writedata(boardPtr, adcpar->gain);
        dt2801_writedata(boardPtr, adcpar->startch);
        dt2801_writedata(boardPtr, adcpar->endch);
        dt2801_writedata2(boardPtr, adcpar->conv);

        stat = dt2801_checkforerror(boardPtr);
        if (stat!=0) {
                return stat;
        }

        adcp->last = 0;
        adcp->gain = adcpar->gain;
        adcp->startch = adcpar->startch;
        adcp->endch = adcpar->endch;
        adcp->conv = adcpar->conv;
        adcp->cnt = 0;
        
	/*
	stat = allocate_mmapadc_buffer(adcp, adcp->conv*sizeof(short));
	if (stat != 0) {
		return stat;
	}
	*/
	stat = allocate_adc_buffer(adcp, adcp->conv*sizeof(short));
	if (stat != 0) {
		return stat;
	}
	
        return 0;
}

static int dt2801_adc_ioctl_a(struct inode *inode, struct file *filp,
			      unsigned int cmd, unsigned long arg)
{
	ADCP* adcp = (ADCP*) filp->private_data;
	Dt2801_t* boardPtr = adcp->board;
	ADC_state* adc_cf = &boardPtr->adc_state;

        switch (cmd) {
        case DT2801_IOCTL_ADC_GET_POLARITY: {
                int stat;
                
                stat = verify_area(VERIFY_WRITE, (char *)arg, sizeof(int));
                if (stat != 0) {
                        return stat;
                }
                put_user(adc_cf->polarity, (int*)arg);
        }
        break;
	case DT2801_IOCTL_ADC_SET_GAIN: {
		int gain = (int) arg;
		adcp->gain = gain;
	}
	break;
	case DT2801_IOCTL_ADC_SETEXTTRIGGER: {
		int exttrig = (int) arg;
		adcp->exttrigger = exttrig;
	}
	break;
        default: {
                return dt2801_do_ioctl(inode, filp, cmd, arg, boardPtr);
        }
        }

        return 0;
}

static int dt2801_adc_ioctl(struct inode *inode, struct file *filp,
                            unsigned int cmd, unsigned long arg)
{
	ADCP* adcp = (ADCP*) filp->private_data;
	Dt2801_t* boardPtr = adcp->board;
	int stat;

        switch (cmd) {
        case DT2801_IOCTL_ADC_SET_PARAMS: {
                Dt2801_ADCP_t uadcp;

                memcpy_fromfs(&uadcp, (char*)arg, sizeof(uadcp));
                adcp->startch = uadcp.startch;
                adcp->endch = uadcp.endch;
                adcp->gain = uadcp.gain;
                adcp->conv = uadcp.conv;
                stat = dt2801_set_adc_params(adcp, &uadcp);
		clear_adc_buffer(adcp);
		return stat;
        }
        break;
        
        /* ioctl SET_ADC_PARAMS has to be called before every call to
           BLOCKSTART */
        case DT2801_IOCTL_ADC_BLOCKSTART: {
                if (boardPtr->fake != 0) {
                        /* Start timer to fill the buffer */
                        if (boardPtr->running == 1) {
                                printk("dt2801: can't start, still running\n");
                                return -1;
                        }
                        clear_adc_buffer(adcp);
                        adcp->cnt = 0;
                
                        boardPtr->fake_counter = 0;
			boardPtr->running = 1;
			boardPtr->stop_req = 0;
                        adc_start_timer(adcp);
                } else {
                        clear_adc_buffer(adcp);
                        adcp->cnt = 0;

                        /* no 'printk's within cli()/sti() sections */
                        cli();
                        disable_dma(boardPtr->dmachannel);
                        clear_dma_ff( boardPtr->dmachannel );
                        set_dma_addr( boardPtr->dmachannel, 
                                      (unsigned long)adcp->dmabuf );
                        set_dma_count( boardPtr->dmachannel,
                                       adcp->dmabufsize );
                        set_dma_mode( boardPtr->dmachannel, DMA_MODE_READ );
                        enable_dma(boardPtr->dmachannel);
                        sti();
                        
                        if (adcp->exttrigger == 1) {
                                dt2801_writecommand(boardPtr,
                                                    DT_C_READ_AD | DT_MOD_DMA
                                                    | DT_MOD_EXTTRIG);
                        } else {
                                dt2801_writecommand(boardPtr,
                                                    DT_C_READ_AD | DT_MOD_DMA);
                        }
                        if (dt2801_checkforerror(boardPtr) != 0) {
                                printk("dt2801: blockstart-dma not ok\n");
                                return -1;
                        }
                        adcp->running = 1;
                }
                
        }
        break;
        case DT2801_IOCTL_ADC_BLOCKSTOP: {
		dt2801_do_stop(boardPtr);
		/* Stopping a command causes the error-flag to be set,
		   will have to clear it !! */
		dt2801_clearerror(boardPtr);
		if (boardPtr->fake == 1) {
			boardPtr->stop_req = 1;
		} else {
			disable_dma(boardPtr->dmachannel);
		}
	}
        break;
        case DT2801_IOCTL_ADC_RESIDUE: {
                int stat, residue;
                
                stat = verify_area(VERIFY_WRITE, (char *)arg, sizeof(int));
                
                if ( stat != 0) {
                        return stat;
                }
                
		cli();
                residue = get_dma_residue(boardPtr->dmachannel);
		sti();
                
                put_user(residue, (int*)arg);
                
        }
        break;
	case DT2801_IOCTL_ADC_SETEXTCLOCK: {
		int extcl = (int) arg;
		adcp->extclock = extcl;
	}
	break;
	case DT2801_IOCTL_ADC_SETCONTINOUS: {
		int cont_flag;
		cont_flag = (int) arg;
	
		adcp->continous = cont_flag;
	}
	break;
        default: {
                return dt2801_adc_ioctl_a(inode, filp, cmd, arg);
        }
	}

    
        return 0;
}


static int adc_use_residue = 0;

static int adc_read_block(struct inode *inode, struct file *filp,
                          char* buffer, int count)
{
	ADCP* adcp = (ADCP*) filp->private_data;
	Dt2801_t* boardPtr = adcp->board;

        int residue;
        int d, stat;


        if (adcp->last == adcp->conv) {
                return -(EIO); /* eof ? */
        }
                
        if (boardPtr->fake != 0) {
                adcp->currnt = get_currnt(adcp);
        } else {
		if (adc_use_residue == 1) {
			cli();
			residue = get_dma_residue(boardPtr->dmachannel);
			/* only transfer complete 2-byte values  */
			adcp->currnt = adcp->conv - ((residue+1)/sizeof(short));
			sti();
		} else {
			adcp->currnt = get_currnt(adcp);
		}
        }
        
        /* no data has arrived yet, generate error */
        if (adcp->currnt  == 0) {
                return -1;
        }
        
        
        d = adcp->currnt - adcp->last;
        if (d > count) {
                d = count;
        }

	/*printk("c l d %d  %d  %d\n", adcp->currnt, adcp->last, d);*/
         
        if (d < 0) {
		printk("d <0\n");
                return -(EINVAL);
        }
        
        if ((stat = verify_area(VERIFY_WRITE, buffer, d*sizeof(short))) != 0) {
                return stat;
        }
        
        memcpy_tofs(buffer, adcp->dmabuf+(adcp->last), d*sizeof(short));
        
        adcp->last+=d;
        
        return d*sizeof(short);
}

static int dt2801_adc_mmap(struct inode *inode, struct file *filp,
			   struct vm_area_struct *vma)
{
	ADCP* adcp = (ADCP*) filp->private_data;

	printk("dt2801: mmap\n");
        if (adcp->dmabuf == NULL) { /* nothing to map */
		return -1;
        }

	printk("dt2801: offset = %lx\n", vma->vm_offset);

	if (remap_page_range(vma_get_start(vma),
			     virt_to_phys(adcp->dmabuf),/*XXX fake-mode?*/
			     vma_get_end(vma) - vma_get_start(vma),
			     vma_get_page_prot(vma))) {
		printk("dt2801: adc: remap bad\n");
		return -EAGAIN;
	}

	vma_set_inode(vma, inode);
	inode_inc_count(inode);

	printk("dt2801: adc remap ok\n");

	return 0;
}

int dt2801_adc_reset(Dt2801_t* boardPtr)
{
	ADC_state* adc_cf = &boardPtr->adc_state;

        adc_cf->gain  = -1;
        adc_cf->startch  = -1;
        adc_cf->endch  = -1;
        adc_cf->conv  = -1;

        return 0;
}

/* May be called even if 'init' has not  !!!!!! */
int dt2801_adc_cleanup(Dt2801_t* boardPtr)
{

        return 0;
}

/* May be given as parameters at load-time */
static char* adc_polarity[] = {
	"unipolar", "unipolar", "unipolar", "unipolar"
};
static char* adc_inputmode[] = {
	"differential", "differential", "differential", "differential"
};

void dt2801_adc_info(Dt2801_t* boardPtr)
{
	ADC_state* adc_cf = &boardPtr->adc_state;
        int bid = boardPtr->boardid;

	printk("dt2801:(%d) ADC: %s, %s\n", bid,
	       adc_polarity[bid], adc_inputmode[bid]);

}

int dt2801_adc_init(Dt2801_t* boardPtr)
{
	ADC_state* adc_cf = &boardPtr->adc_state;
        int bid = boardPtr->boardid;

        adc_cf->bits  = 12;

        if (!strcmp(adc_polarity[bid], "unipolar")) {
		adc_cf->polarity  = DT2801_ADC_UNIPOLAR;
	} else if (!strcmp(adc_polarity[bid], "bipolar")) {
		adc_cf->polarity  = DT2801_ADC_BIPOLAR;
	} else {
		printk("dt2801: invalid value for 'polarity': %s\n",
		       adc_polarity[bid]);
		return -(EINVAL);
	}

        if (!strcmp(adc_inputmode[bid], "differential")) {
		adc_cf->inputmode  = DT2801_ADC_DIFFERENTIAL;
	} else if (!strcmp(adc_inputmode[bid], "single-ended")) {
		adc_cf->inputmode  = DT2801_ADC_SINGLE_ENDED;
	} else {
		printk("dt2801: invalid value for 'inputmode': %s\n",
		       adc_inputmode[bid]);
		return -(EINVAL);
	}

        if (adc_cf->inputmode == DT2801_ADC_DIFFERENTIAL) {
		adc_cf->nchannels  = 8;
	} else {
		adc_cf->nchannels  = 16;
	}

        return 0;
}


static struct file_operations adc_fops_imm = {
        NULL,                /* seek */
        adc_read_imm,        /* read */
        NULL,                /* write */
        NULL,                /* readdir */
        NULL,                /* select */
        dt2801_adc_ioctl_a,  /* ioctl */
        NULL,                /* mmap */
        dt2801_adc_open,     /* open */
        dt2801_adc_release   /* release */
};

static struct file_operations adc_fops_block = {
        NULL,                /* seek */
        adc_read_block,      /* read */
        NULL,                /* write */
        NULL,                /* readdir */
        NULL,                /* select */
        dt2801_adc_ioctl,    /* ioctl */
        dt2801_adc_mmap,     /* mmap */
        dt2801_adc_open,     /* open */
        dt2801_adc_release   /* release */
};

int dt2801_adc_open(struct inode *inode, struct file *filp)
{
	int minor = MINOR(inode->i_rdev) & 0xc;
	int channel;
	ADCP* adcp;
        Dt2801_t* boardPtr = filp->private_data;

	DT_PRINTKF("minor = %d\n", minor);

	if ((minor < 0 || minor > 15) && minor != DT2801_ADCX) {
		return -(ENODEV);
	}
        if ( (minor & 0x1f) == DT2801_ADCX) {
		channel = -1;
	} else {
		channel = minor & 0xf;
                if (channel < 0 || channel >= boardPtr->adc_state.nchannels) {
			return -(ENODEV);
		}
	}

	DT_PRINTKF("channel = %d\n", channel);

        if (channel == -1 ) {
		/* non-immediate (block-commands) */
		filp->f_op = &adc_fops_block;
	} else {
		/* immediate */
		filp->f_op = &adc_fops_imm;
	}

	adcp = (ADCP*) kmalloc(sizeof(ADCP), GFP_KERNEL);
	if (adcp == NULL) {
		return -(ENOMEM);
	}

        adcp->board = boardPtr;

	adcp->exttrigger = 0;
	adcp->extclock = 0;
	adcp->continous = 0;

        adcp->startch = 0;
        adcp->endch = 0;
        adcp->gain = 0;
        adcp->conv = 0;

	adcp->last = 0;
	adcp->currnt = 0;
	adcp->running = 0;
	adcp->cnt = 0;

	adcp->dmabufsize = 0;
	adcp->dmabuf = NULL;

	filp->private_data = adcp;

        return 0;
}
#endif

/*
 * Device Driver for DataTranslation DT2801
 *
 * Digital-Analog-Converter subsystem
 *
 *
 */


/* Available: 
      DAC0/1 immediate mode */

/* not Available: 
      block modes */

/* I really dont't know how to integrate the mode, where
   both DACs work together. As its own device it is not like
   _one_ DAC, because it has two channels that are completley
   independent, except that their outputs are set simultanously.

   On the other hand, to make a block-mode (with DMA) work, data
   for both channels must be given.

   Has to wait, and is currently disabled. */
   
  
#if 0

/* How should that be handled ?
   I think we need all the data in a buffer before we write it
   (with DMA). How to deal with continous mode ? */
static int dac_write_block(struct inode *inode, struct file *filp,
			    const char* buffer, int count)
{
	return 0;
}

static int dac_write(struct inode *inode, struct file *filp,
			    const char* buffer, int count)
{

	DACP* dacp = (DACP*) filp->private_data;

	switch (dacp->mode) {
	case DT2801_DAC_IMMEDIATE: {
		return dac_write_imm(inode, filp, buffer, count);
	} break;

	case DT2801_DAC_BLOCK: {
		return dac_write_block(inode, filp, buffer, count);
	}

	}

	return 0;
}

static int dac_setparams(Dt2801_t* boardPtr, int dacnr, int conv)
{
	DAC_state* dac_cf = &boardPtr->dac_state;

	dt2801_writecommand(boardPtr, DT_C_SET_DA);
	dt2801_writedata(boardPtr, dacnr);
	dt2801_writedata2(boardPtr, conv);

	dac_cf->dacnr = dacnr;
	dac_cf->conv = conv;
	
	return 0;
}

static int dac_ioctl(struct inode *inode, struct file *filp,
			    unsigned int cmd, unsigned long arg)
{
	DACP* dacp = (DACP*) filp->private_data;
	Dt2801_t* boardPtr = dacp->board;
	DAC_state* dac_cf = &boardPtr->dac_state;

	switch(cmd) {
	case DT2801_IOCTL_DAC_SET_PARAMS: {
		Dt2801_DACP_t dacparams;
                int dacnr, conv;

                dacnr = dacp->dacnr;
                conv = 0; /* XXX */

		memcpy_fromfs(&dacp, (char*) arg, sizeof(dacparams));

		dac_setparams(boardPtr, dacnr, conv);

		return dt2801_checkforerror(boardPtr);
	}
	break;
	case DT2801_IOCTL_DAC_GET_POLARITY: {
                int dacnr;
                int stat;
                
                dacnr = dacp->dacnr;
                stat = verify_area(VERIFY_WRITE, (char *)arg, sizeof(int));
                if (stat != 0) {
                        return stat;
                }
                put_user(dac_cf->dacs[dacnr].polarity, (int *)arg);

		return 0;
	}
	break;
	case DT2801_IOCTL_DAC_GET_FULLSCALE: {
                int dacnr;
                int stat;

		dacnr = dacp->dacnr;
                stat = verify_area(VERIFY_WRITE, (char *)arg, sizeof(int));
                if (stat != 0) {
                        return stat;
                }
                put_user(dac_cf->dacs[dacnr].fullscale, (int *)arg);
		return 0;
	}
	break;	case DT2801_IOCTL_DAC_SET_MODE: {

                dacp->mode = (int)arg;
		return 0;
	}
	break;
	default:
		return dt2801_do_ioctl(inode, filp, cmd, arg, boardPtr);
	}
	
	return 0;
}

int dt2801_dac_reset(Dt2801_t* boardPtr)
{
	DAC_state* dac_cf = &boardPtr->dac_state;

        /* Both DAC's are set to 0 after power on or reset, which
	   gives 0V for unipolar and -fullscale for bipolar
	   configuration. */

        dac_cf->dacnr  = -1;
        dac_cf->conv  = -1;

        /* Maybe set them to 0V for bipolar ? */
	/*
	  data = 2048; nr = 0;
	  dt2801_writecommand(DT_C_WRITE_DAIM);
	  dt2801_writedata(nr);
	  dt2801_writedata2(data);
	  data = 2048; nr = 1;
	  dt2801_writecommand(DT_C_WRITE_DAIM);
	  dt2801_writedata(nr);
	  dt2801_writedata2(data);
	*/

	return 0;
}

/* May be called even if 'init' has not  !!!!!! */
int dt2801_dac_cleanup(Dt2801_t* boardPtr)
{
	return 0;
}


static int dac_parse_polarity(char *str)
{
	int polarity;

	if (!strcmp(str, "unipolar")) {
		polarity  = DT2801_DAC_UNIPOLAR;
	} else if (!strcmp(str, "bipolar")) {
		polarity  = DT2801_DAC_BIPOLAR;
	} else {
		return -(EINVAL);
	}

        return polarity;
}

/* Is there a scanf() in the kernel ? */
/* Is there anything special about using floatingpoint variables 
   in the kernel ? */
/* How does insmod deal with more arguments than array elements ?*/

/* ! modules-2.0.0 has bug that arrays of (char*) are not properly
   set. Seems to be fixed in 2.1.8 and above. 2.1.13 is ok, checked
   the source */
#define DT2801_DAC_MAX_FSNAMES 3
static char* dac_fs_names[DT2801_DAC_MAX_FSNAMES] = {"10.0", "5.0", "2.5"};

static int dac_parse_fullscale(char *str)
{
        int i;

        for (i=0; i<DT2801_DAC_MAX_FSNAMES; i++) {
		if (!strcmp(str, dac_fs_names[i])) {
			return i;
		}
	}
	return -(EINVAL);
}


/* May be given as parameters at load-time */
static char* dac_fullscale[] =  {
	"10.0", "10.0",
	"2.5", "2.5",
	"5.0", "5.0",
	"2.5", "2.5"
};
static char* dac_polarity[] =  {
	"bipolar", "bipolar",
	"bipolar", "bipolar",
	"bipolar", "unipolar",
	"unipolar", "bipolar",
};

void dt2801_dac_info(Dt2801_t* boardPtr)
{
	DAC_state* dac_cf = &boardPtr->dac_state;
        int bid = boardPtr->boardid;
        int i;

        for (i=0; i<DT2801_NDACS; i++) {
		printk("dt2801:(%d) DAC%d: bits(%d), %s, %s\n", bid, i,
		       dac_cf->dacs[i].bits,
		       dac_polarity[2*bid+i], dac_fullscale[2*bid+i]);
	}
}

int dt2801_dac_init(Dt2801_t* boardPtr)
{
	DAC_state* dac_cf = &boardPtr->dac_state;
        int bid = boardPtr->boardid;

        int i;
        DAC_t* dac;

        dac_cf->conv = 0;
        dac_cf->dacnr = -1;

        for (i=0; i<DT2801_NDACS; i++) {

		dac = &dac_cf->dacs[i];

		dac->polarity  = dac_parse_polarity(dac_polarity[2*bid+i]);
		if (dac->polarity < 0) {
			printk("dt2801: invalid value for 'dac_polarity': "
			       "%s\n", dac_polarity[2*bid+i]);
			return -(EINVAL);
		}

		dac->fullscale = dac_parse_fullscale(dac_fullscale[2*bid+i]);
		if (dac->fullscale < 0) {
			printk("dt2801: invalid value for 'dac_fullscale': "
			       "%s\n", dac_fullscale[2*bid+i]);
			return -(EINVAL);
		}

		if (dac->fullscale == 2 &&
		    dac->polarity == DT2801_DAC_UNIPOLAR) {
			printk("dt2801: unipolar/%sV not available\n",
			       dac_fs_names[dac->fullscale]);
			return -(EINVAL);
		}

		if (boardPtr->boardid == DT2801_T_2808) {
			dac_cf->dacs[i].bits = 8;
		} else {
			dac_cf->dacs[i].bits = 12;
		}
		dac_cf->dacs[i].usecount = 0;
	}


	return 0;
}



int dt2801_dac_open(struct inode *inode, struct file *filp)
{
	int minor = MINOR(inode->i_rdev) & 0xc;
        int dacnr;
	DACP* dacp;

	Dt2801_t* boardPtr = (Dt2801_t*) filp->private_data;
	DAC_state* dac_cf = &boardPtr->dac_state;

	DT_PRINTKF("minor = %d\n", minor);

        if (minor < 0 || minor >= 2) { /* XXX  > 2*/
		return -(ENODEV);
        }

        dacnr = minor & 0x3;
	DT_PRINTKF("dacnr = %d\n", dacnr);
        if (dacnr < 0 || dacnr > 2) {
		printk("dt2801: dac_open: invalid dacnr = %d.\n", dacnr);
		return -(EINVAL);
        }

        if (dacnr == 0 || dacnr == 1) {
		if (dac_cf->dacs[dacnr].usecount != 0) {
			printk("dt2801: dac_open: "
			       "dacnr = %d not free.\n", dacnr);
			return -(EBUSY);
		}
		dac_cf->dacs[dacnr].usecount++;

	} else if (dacnr == 2) {
		if (dac_cf->dacs[0].usecount != 0 ||
		    dac_cf->dacs[1].usecount != 0) {
			printk("dt2801: dac_open: dac 0+1 not free.\n");
			return -(EBUSY);
		}
		dac_cf->dacs[0].usecount++;
		dac_cf->dacs[1].usecount++;
	} else {
		printk("dt2801: dac_open: invalid dacnr = %d.\n", dacnr);
		return -(EINVAL);
	}

	dacp = (DACP*) kmalloc(sizeof(DACP), GFP_KERNEL);
	if (dacp == NULL) {
		return -(ENOMEM);
	}
        dacp->dacnr = dacnr;
        dacp->mode = DT2801_DAC_IMMEDIATE;
        dacp->exttrigger = 0;
	filp->private_data = dacp;
	filp->f_op = &dac_fops;

	return 0;
}
#endif


/*
 * Device Driver for DataTranslation DT2801
 *
 * Digital-I/O subsystem
 *
 */

#if 0

static int set_port_mode(Dt2801_t* boardPtr, int port, int mode) 
{
        DIO_state* dio_cf = &boardPtr->dio_state;

	if (port < 0 || port > dio_cf->nports) {
		printk("dt2801: set_port_mode: invalid port #%d\n", port);
		return -(EINVAL);
	}

        switch (mode) {
        case DT2801_DIO_PORTIN:
		dt2801_writecommand(boardPtr, DT_C_SET_DIGIN);
		dt2801_writedata(boardPtr, port);
                dio_cf->ports[port].mode = DT2801_DIO_PORTIN;
		break;
        case DT2801_DIO_PORTOUT:
		dt2801_writecommand(boardPtr, DT_C_SET_DIGOUT);
		dt2801_writedata(boardPtr, port);
                dio_cf->ports[port].mode = DT2801_DIO_PORTOUT;
		break;
        default:
		printk("dt2801: set_port_mode: invalid mode %d\n", mode);
		return -(EINVAL);
	}

	return 0;
}
 
static int do_write(Dt2801_t* boardPtr, int portnr) 
{
        DIO_state* dio_cf = &boardPtr->dio_state;
	DIO_port* port;
	int i, databyte;

	switch (portnr) {
        case DT2801_DIO0:
        case DT2801_DIO1: {
	}
        case DT2801_DIO2: {
	}
	}
        for (i=0; i<dio_cf->nports; i++) {
		port = &dio_cf->ports[i];
		if (port->mode == DT2801_DIO_PORTOUT) {
			databyte = port->lines;
			dt2801_writecommand(boardPtr, DT_C_WRITE_DIG);
			dt2801_writedata(boardPtr, i);
			dt2801_writedata(boardPtr, databyte);
		} else {
			printk("dt2801: do_write: port %d not writable\n", i);
			return -(EINVAL);
		}
	}
	return 0;
}

static int dio_read(struct inode *inode, struct file *filp,
			   char* buffer, int count)
{
	DIOP* diop = (DIOP*) filp->private_data;
	int minor = MINOR(inode->i_rdev);
	int port = diop->port;

	printk("dt2801: dio_read: minor = %d  port = %d.\n", minor, port);


	return 0;
}

static int dio_read2(struct inode *inode, struct file *filp,
		     char* buffer, int count)
{
	DIOP* diop = (DIOP*) filp->private_data;
	int minor = MINOR(inode->i_rdev);
	int port = diop->port;

	printk("dt2801: dio_read2: minor = %d  port = %d.\n", minor, port);


	return 0;
}

static int dio_write(struct inode *inode, struct file *filp,
		     const char* buffer, int count)
{
	DIOP* diop = (DIOP*) filp->private_data;
	Dt2801_t* boardPtr = diop->board;
        int databyte = 0;


	int minor = MINOR(inode->i_rdev);
	int port = diop->port;

	printk("dt2801: dio_write: minor = %d  port = %d.\n", minor, port);

	dt2801_writecommand(boardPtr, DT_C_WRITE_DIG);
	dt2801_writedata(boardPtr, port);
	dt2801_writedata(boardPtr, databyte);

	return count;
}

static int dio_write2(struct inode *inode, struct file *filp,
		      const char* buffer, int count)
{
	DIOP* diop = (DIOP*) filp->private_data;
	Dt2801_t* boardPtr = diop->board;
        int databyte = 0;


	int minor = MINOR(inode->i_rdev);
	int port = diop->port;

	printk("dt2801: dio_write2: minor = %d  port = %d.\n", minor, port);

	dt2801_writecommand(boardPtr, DT_C_WRITE_DIG);
	dt2801_writedata(boardPtr, port);
	dt2801_writedata2(boardPtr, databyte);

	return count;
}

static int dio_ioctl(struct inode *inode, struct file *filp,
			    unsigned int cmd, unsigned long arg)
{
	DIOP* diop = (DIOP*) filp->private_data;
	Dt2801_t* boardPtr = diop->board;

	int minor = MINOR(inode->i_rdev);
	int port = diop->port;

	printk("dt2801: dio_ioctl: minor = %d  port = %d.\n", minor, port);

	switch (cmd) {
	default:
		return dt2801_do_ioctl(inode, filp, cmd, arg, boardPtr);
	}
	return 0;
}

static int dio_ioctl_g(struct inode *inode, struct file *filp,
			    unsigned int cmd, unsigned long arg)
{
	DIOP* diop = (DIOP*) filp->private_data;
	Dt2801_t* boardPtr = diop->board;

	int minor = MINOR(inode->i_rdev);
	int port = diop->port;

	printk("dt2801: dio_ioctl_g: minor = %d  port = %d.\n", minor, port);

	switch (cmd) {
	case DT2801_IOCTL_DIO_SET_LINES: {
	}
	break;
	default:
		return dt2801_do_ioctl(inode, filp, cmd, arg, boardPtr);
	}
	return 0;
}



/* After board-reset or power-up,
   set both ports to INPUT */
int dt2801_dio_reset(Dt2801_t* boardPtr)
{
        DIO_state* dio_cf = &boardPtr->dio_state;
        int i;

        for (i=0; i<dio_cf->nports; i++) {
		dio_cf->ports[i].mode = DT2801_DIO_PORTIN;
	}

	return 0;
}

/* Called by 'cleanup_module'
   May be called even if 'init' has not  !!!!!! */
int dt2801_dio_cleanup(Dt2801_t* boardPtr)
{

	return 0;
}

void dt2801_dio_info(Dt2801_t* boardPtr)
{
        DIO_state* dio_cf = &boardPtr->dio_state;
        int bid = boardPtr->boardid;

	printk("dt2801:(%d) DIO: %d ports\n", bid, dio_cf->nports);
}

int dt2801_dio_init(Dt2801_t* boardPtr)
{
        DIO_state* dio_cf = &boardPtr->dio_state;
        int i;

        dio_cf->usecount = 0;
        dio_cf->nports = DT2801_DIO_NPORTS;

        for (i=0; i<dio_cf->nports; i++) {
		dio_cf->ports[i].usecount = 0;
		dio_cf->ports[i].mode = DT2801_DIO_PORTUNDEF;
		dio_cf->ports[i].lines = 0;
	}

	return 0;
}


/* A D-I/O port can be opened either for reading or writing
   but not read-write. */
int dt2801_dio_open(struct inode *inode, struct file *filp)
{
        int stat = 0;
	int minor = MINOR(inode->i_rdev) & 0xc;
	int mode;
        int i;

	Dt2801_t* boardPtr = filp->private_data;
        DIO_state* dio_cf = &boardPtr->dio_state;

	DIOP* diop;

	printk("dt2801: dio_open: minor = % d.\n", minor);

	diop = (DIOP*) kmalloc(sizeof(DIOP), GFP_KERNEL);
	if (diop == NULL) {
		return -(ENOMEM);
	}

        diop->board = boardPtr;

	mode = filp->f_flags & O_ACCMODE;

	if (mode == O_RDWR) {
		stat = -(EPERM);
		goto fail1;
	}

	switch (minor) {
        case DT2801_DIO0:
        case DT2801_DIO1: {
		int port = minor & 0x3;

		printk("dt2801: dio_open: port = % d.\n", port);

		if (mode == O_RDONLY) {
			if (dio_cf->ports[port].mode != DT2801_DIO_PORTIN) {
				if (dio_cf->ports[port].usecount != 0) {
					stat = -(EBUSY);
					goto fail1;
				}
				stat = set_port_mode(boardPtr, port,
						     DT2801_DIO_PORTIN);
				if (stat <0) { goto fail1; }
			}
			diop->mode = DT2801_DIO_PORTIN;
			filp->f_op = &dio_fops_r;
		} else { /*O_WRONLY */
			if (dio_cf->ports[port].usecount != 0) {
				stat = -(EBUSY);
				goto fail1;
			}
			stat = set_port_mode(boardPtr, port, DT2801_DIO_PORTOUT);
			if (stat <0) { goto fail1; }
			diop->mode = DT2801_DIO_PORTOUT;
			filp->f_op = &dio_fops_w;
		} 
		dio_cf->ports[port].usecount++;
		
		diop->port = port;
		break;
        }
        case DT2801_DIO2: {
		int port = minor & 0x3; 

		if (mode == O_RDONLY) {
			for (i=0; i<dio_cf->nports; i++) {
				if (dio_cf->ports[i].mode != DT2801_DIO_PORTIN) {
				    if (dio_cf->ports[i].usecount != 0) {
					stat = -(EBUSY);
					goto fail1;
				    }
				    stat = set_port_mode(boardPtr, i,
							 DT2801_DIO_PORTIN);
				    if (stat <0) { goto fail1; }
				}
			}
			diop->mode = DT2801_DIO_PORTIN;
			filp->f_op = &dio_fops_r2;
		} else { /*O_WRONLY */
			for (i=0; i<dio_cf->nports; i++) {
				if (dio_cf->ports[i].usecount != 0) {
					stat = -(EBUSY);
					goto fail1;
				}
			}
			for (i=0; i<dio_cf->nports; i++) {
				stat = set_port_mode(boardPtr, i,
						     DT2801_DIO_PORTOUT);
				if (stat <0) { goto fail1; }
			}
			diop->mode = DT2801_DIO_PORTOUT;
			filp->f_op = &dio_fops_w2;
		} 
		dio_cf->ports[0].usecount++;
		dio_cf->ports[1].usecount++;
		
		diop->port = port;

		break;
	}
        case DT2801_DIOX: {
		printk("dt2801: dio_open: generic device not implemented.\n");
		stat = -(ENODEV);
		goto fail1;

		/* filp->f_op->ioctl = dio_ioctl_g; */

		break;
	}
        default: {
		int line = minor & 0xf;
		int port;

		printk("dt2801: dio_open: line = % d.\n", line);
		if (line <0 || line > dio_cf->nports*DT2801_DIO_PORTSIZE) {
			stat = -(ENODEV);
			goto fail1;
		}
		port = line/DT2801_DIO_PORTSIZE;
		printk("dt2801: dio_open: port = % d.\n", port);
		if (mode == O_RDONLY) {
			if (dio_cf->ports[port].mode != DT2801_DIO_PORTIN) {
				if (dio_cf->ports[port].usecount != 0) {
					stat = -(EBUSY);
					goto fail1;
				}
				stat = set_port_mode(boardPtr, port,
						     DT2801_DIO_PORTIN);
				if (stat <0) { goto fail1; }
			}
			diop->mode = DT2801_DIO_PORTIN;
			filp->f_op = &dio_fops_r2; /* XXX */
		} else { /*O_WRONLY */
			if (dio_cf->ports[port].usecount != 0) {
				stat = -(EBUSY);
				goto fail1;
			}
			stat = set_port_mode(boardPtr, port, DT2801_DIO_PORTOUT);
			if (stat <0) { goto fail1; }
			diop->mode = DT2801_DIO_PORTOUT;
			filp->f_op = &dio_fops_w2; /* XXX */
		}

		dio_cf->ports[port].usecount++;
		
		diop->port = port;
	}
	}

	filp->private_data = diop;

	for (i=0; i<dio_cf->nports; i++) {
		printk("dt2801: dio_usecount[%d] = %d\n.", 
		       i, dio_cf->ports[i].usecount);
	}
	return 0;

fail1:
	kfree(diop);
        return stat;
}
#endif


/*
 * Device Driver for DataTranslation DT2801
 *
 * Linux kernel 2.0.x
 *
 */


#if 0

static Dt2801_t* boardlist;
static nboards = 0;
/* An element corresponds to a board 
   It will be initialized with pointers into 'boardlist' */
static Dt2801_t* boards[DT2801_BOARDS_MAX] = {NULL, NULL, NULL, NULL};



static void dt2801_timer_interrupt(unsigned long bd)
{
	Dt2801_t* boardPtr = boards[bd];
	if (!boardPtr->dt_timer_int)  return;

	wake_up_interruptible(&boardPtr->waitq);
}

static void dt2801_start_timer(Dt2801_t* boardPtr)
{
	init_timer(&boardPtr->dt_timer_list);
	boardPtr->dt_timer_list.expires = jiffies + DT2801_TIMERCYCLES;
	boardPtr->dt_timer_list.function = dt2801_timer_interrupt;
	boardPtr->dt_timer_list.data = boardPtr->boardid;
	boardPtr->dt_timer_int = 1;
	add_timer(&boardPtr->dt_timer_list);
}

/* Stop board, and do a dummy read to clear DataOutReady-bit */
int dt2801_do_stop(Dt2801_t* boardPtr)
{
	int stat, ignored;

	if (boardPtr->fake == 1) {
		return 0;
	}

	stat = inb_p(dt_status(boardPtr));
	/* printk("dt2801: before stop: status = 0x%0x\n", stat); */
	outb_p(DT_C_STOP, dt_command(boardPtr));
	/* sleep a bit ? Wait for ready flag. */

		dt2801_wait(boardPtr);
	ignored = inb_p(dt_data_in(boardPtr));

	stat = inb_p(dt_status(boardPtr));
	/* printk("dt2801: after stop: status = 0x%0x\n", stat); */
		dt2801_wait(boardPtr);
	stat = inb_p(dt_status(boardPtr));
	/* printk("dt2801: after stop: status = 0x%0x\n", stat); */

	return 0;
}




/* Check composite-error-flag. If set read error-register and clear
 */

/* Maybe reset board etc, if we fail to determine the error-code */
int dt2801_checkforerror(Dt2801_t* boardPtr)
{
	int stat;
	int err;

	if (boardPtr->fake == 1) {
		return 0;
	}

	stat = inb_p(dt_status(boardPtr));

        if (stat & DT_S_COMPOSITE_ERROR) {

		/* printk("dt2801: checkforerror: status = 0x%0x.\n", stat); */
		stat = dt2801_writecommand(boardPtr, DT_C_READ_ERRREG);
                if (stat != 0) {
			printk("dt2801: checkforerror: failed to read"
			       " error-reg\n");
			printk("dt2801: reload the driver.\n");
		}
		stat = dt2801_readdata2(boardPtr, &err);
                if (stat != 0) {
			printk("dt2801: checkforerror: failed to read"
			       " error-data\n");
			printk("dt2801: reload the driver.\n");
		}
		printk("dt2801: Error #%d.\n", err);

		/* Clear error-register and composite-error flag */
		stat = dt2801_writecommand(boardPtr, DT_C_CLEAR_ERR);
		if (stat != 0) {
			printk("dt2801: checkforerror: clear-error failed\n");
		}
			
		return err;
	}

        return 0;
}

int dt2801_wait_cmd_completed(Dt2801_t* boardPtr) {
	int stat;

	if (boardPtr->fake == 1) {
		return 0;
	}

	stat = inb_p(dt_status(boardPtr));
	/* printk("dt2801: cmd-1: status = 0x%0x\n", stat); */
	if ( (stat & DT_S_READY) && !(stat & DT_S_COMPOSITE_ERROR)) {
		return 0;
	}
	dt2801_wait(boardPtr);
	stat = inb_p(dt_status(boardPtr));
	/* printk("dt2801: cmd-2: status = 0x%0x\n", stat); */
	if ( (stat & DT_S_READY) && !(stat & DT_S_COMPOSITE_ERROR)) {
		return 0;
	}
	printk("dt2801: cmd not successfully completed\n");

	return -1;
}

int dt2801_clearerror(Dt2801_t* boardPtr) {
	int stat;

	if (boardPtr->fake == 1) {
		return 0;
	}

	stat = inb_p(dt_status(boardPtr));
	/* printk("dt2801: clerrerr-1: status = 0x%0x\n", stat); */
	stat = dt2801_writecommand(boardPtr, DT_C_CLEAR_ERR);
        if (stat != 0) {
		dt2801_checkforerror(boardPtr);
		return -1;
	}
	stat = dt2801_wait_cmd_completed(boardPtr);
        if (stat != 0) {
		dt2801_checkforerror(boardPtr);
		return -1;
	}

	return 0;
}

int dt2801_reset(Dt2801_t* boardPtr)
{
	int stat;

	stat = dt2801_do_reset(boardPtr);
        if (stat < 0) {
               printk("dt2801: reset: do_reset failed\n");
        }

	/* defaults after RESET or power-on */
	dt2801_adc_reset(boardPtr);
	dt2801_dac_reset(boardPtr);
	dt2801_dio_reset(boardPtr);

	boardPtr->clock.ticks = 200;  /* 200 ticks =  1ms for DT2801 */

	return 0;
}

static int dt2801_setclock(Dt2801_t* boardPtr, int period)
{
	int stat, err;

	if (boardPtr->fake == 1) {
		return 0;
	}

	stat = inb_p(dt_status(boardPtr));
        /* printk("dt2801: setclock status = 0x%0x\n", stat); */
	if (stat & DT_S_COMPOSITE_ERROR) {
		err =  dt2801_checkforerror(boardPtr);
                printk("dt2801: setclock err = 0x%0x\n", err);
                return -1;
        }
        /* printk("dt2801: setclock %d\n", period); */
	stat = dt2801_writecommand(boardPtr, DT_C_SET_CLOCK);
        if (stat != 0) {
		err =  dt2801_checkforerror(boardPtr);
		return -1;
	}
	dt2801_writedata2(boardPtr, period);
        if (stat != 0) {
		err =  dt2801_checkforerror(boardPtr);
		return -1;
	}

	stat = dt2801_wait_cmd_completed(boardPtr);
        if (stat != 0) {
		err =  dt2801_checkforerror(boardPtr);
		return -1;
	}

	return stat;
}

int dt2801_do_ioctl(struct inode *inode, struct file *filp,
		    unsigned int cmd, unsigned long arg, Dt2801_t* boardPtr)
{

	switch (cmd) {
	case DT2801_IOCTL_STOP: {
		dt2801_do_stop(boardPtr);
	}
	break;

	case DT2801_IOCTL_SETCLOCK: {
		int cl;
		cl = (int) arg;
		
		if ( dt2801_setclock(boardPtr, cl) != 0) {
			return -1;
		}
		boardPtr->clock.ticks = cl;
	}
	break;
		
	default:
		printk("dt2801: unknown ioctl %u.\n", cmd);
		return -EINVAL;
	}
	
	return 0;
}




int dt2801_open(struct inode *inode, struct file *filp)
{
	int stat = 0;
	int minor = MINOR(inode->i_rdev);
	int mode = filp->f_flags & O_ACCMODE;
	int board = DT2801_BOARD(minor);
	Dt2801_t* boardPtr;


	MOD_INC_USE_COUNT;

        if (board <0 || board >= nboards) {
		printk("dt2801_open: no board %d\n", board);
		stat = -(ENODEV);
		MOD_DEC_USE_COUNT;
	        return stat;
	}

	boardPtr = boardlist + board;

	/* stat = inb_p(dt_status(boardPtr));
	   printk("dt2801: open: status = 0x%0x\n", stat); */

        if (boardPtr->firstopen == 1) {
         	if (dt2801_do_reset(boardPtr) < 0) {
	        	printk("dt2801_open: do_reset failed!?\n");
			stat = -(EIO);
			goto fail1;
	        }
		boardPtr->firstopen = 0;
        }


	/* pass the pointer to the board-struct to 'open' */
	filp->private_data = boardPtr;

	if (DT2801_IS_ADC(minor) || DT2801_IS_DAC(minor)) {
		if (DT2801_IS_ADC(minor) && mode == O_RDONLY) {
			stat = dt2801_adc_open(inode, filp);
		} else if (DT2801_IS_DAC(minor) && mode == O_WRONLY) {
			stat = dt2801_dac_open(inode, filp);

		} else {
			printk("dt2801_open: minor = %d, bad AD/DA mode.\n",
			       minor);
			stat = -(EPERM);
		}

	} else if (DT2801_IS_DIO(minor)) {
		stat = dt2801_dio_open(inode, filp);
   
	} else {
		printk("dt2801_open: invalid minor = 0x%02x.\n", minor);
		stat = -(ENODEV);
	}

fail1:
	if (stat < 0) {
		MOD_DEC_USE_COUNT;
	        return stat;
	}
	
        boardPtr->usecount++;
	DT_PRINTKF("%d\n", boardPtr->usecount);

	return stat;

}

struct file_operations dt2801_fops = {
	NULL,            /* seek */
	NULL,            /* read */
	NULL,            /* write */
	NULL,            /* readdir */
	NULL,            /* select */
	dt2801_ioctl,    /* ioctl */
	NULL,            /* mmap */
	dt2801_open,     /* open */
	dt2801_release   /* release */
};



/*
 These can be set from the commandline, when loading the module.
*/

/* 0: normal operation, 1: simulate the hardware */
static int fake[DT2801_BOARDS_MAX] = {0, 0 ,0, 0};

/* 0 will result in a dynamically asigned number */
static int major = DT2801_MAJOR;

/* 0 means: no board */
static int io[DT2801_BOARDS_MAX] = {DT2801_BASEADDR, 0x0 ,0x0, 0x0};
/* -1 means: board  will not use dma */
static int dma[DT2801_BOARDS_MAX] = {3, -1,-1, -1};

int init_module(void)
{
	Dt2801_t *boardPtr;
        int i, err, stat = 0;

	printk("dt2801: DataTranslation-DT2801 driver, %s\n", rcs_revision);

        nboards = 0;
	for (i=0; i<DT2801_BOARDS_MAX; i++) {
                int baseaddr = io[i];
                int dmachannel = dma[i];

		if (baseaddr == 0) break;
		if (baseaddr < 0x2 || baseaddr > 0x3fe) {
			printk("dt2801: invalid base address: %d\n", baseaddr);
			return -(EINVAL);
		}

		if (dmachannel != -1) {
			if (dmachannel<1 || dmachannel > 3) {
				printk("dt2801: invalid dmachannel: %d\n",
				       dmachannel);
				return -(EINVAL);
			}
                }
                nboards++;
	}

	boardlist = (Dt2801_t *)kmalloc( sizeof(Dt2801_t) * nboards,
					    GFP_KERNEL);
	if (boardlist == NULL) {
		return -(ENOMEM);
	}

	for (i=0, boardPtr=boardlist; i<nboards; i++, boardPtr++) {

		boards[i] = boardPtr;

		boardPtr->typeid = 0;
		boardPtr->typename = NULL;

		boardPtr->boardid = i;
		boardPtr->fake = fake[i];
		
		boardPtr->dmachannel = -1;
		boardPtr->baseaddr = 0;

		boardPtr->inuse = 0;
		boardPtr->usecount = 0;

		boardPtr->inerror = 0;
		boardPtr->firstopen = 1;
		boardPtr->status = -1;
		boardPtr->dma_running = 0;
		boardPtr->exttrigger_expected = 0;

		boardPtr->dt_timer_int = 0;
		boardPtr->waitq = NULL;
	}

	for (i=0, boardPtr=boardlist; i<nboards; i++, boardPtr++) {
		int dmachannel = dma[i];
		int baseaddr = io[i];

		if (dmachannel != -1) {
			if ( request_dma(dmachannel, "dt2801") ) {
				printk("dt2801_init: Cannot request DMA %d"
				       " for board %d.\n",
				       i,dmachannel);
				boardPtr->dmachannel = -1;
			} else {
				boardPtr->dmachannel = dmachannel;
			}
		}


		if (check_region(baseaddr, DT2801_IOSIZE) != 0) {
			printk("dt2801: init_module: " 
			       "I/O ports 0x%x to 0x%x already in use.\n",
			       baseaddr, baseaddr + DT2801_IOSIZE-1);
			stat = -(EINVAL);
			goto fail_major;
		}
		request_region(baseaddr, DT2801_IOSIZE, "dt2801");
		boardPtr->baseaddr = baseaddr;
	}

	err = register_chrdev(major, "dt2801", &dt2801_fops);
	if (err < 0) {
		printk("dt2801: init_module: "
		       "Cannot register to major device %d.\n", major);
		goto fail_major;
	}
	if (major == 0) {
		major = err; /* dynamic major # */
	}


	/* Stop to make sure the board can accept the next command */
	for (i=0, boardPtr=boardlist; i<nboards; i++, boardPtr++) {
		dt2801_do_stop(boardPtr); 

		stat = dt2801_do_reset(boardPtr);
		if (stat < 0) {
			printk("dt2801_init: board (%d) do_reset failed!?\n",
			       i);
			goto fail3;
		} else {
			boardPtr->typeid = stat;
			boardPtr->typename = typeid2name(boardPtr->typeid);
		}
	}

	/* Initialize Clock, A/D, D/A, D-I/O */

	for (i=0, boardPtr=boardlist; i<nboards; i++, boardPtr++) {
		boardPtr->clock.min = 2;
		boardPtr->clock.max = 0xffff;

		if (dt2801_adc_init(boardPtr) != 0) {
			printk("dt2801_adc_init: failed\n");
			stat = -1;
			goto error;
		}
		if (dt2801_dac_init(boardPtr) != 0) {
			printk("dt2801_dac_init: failed\n");
			stat = -1;
			goto error;
		}
		if (dt2801_dio_init(boardPtr) != 0) {
			printk("dt2801_dio_init: failed\n");
			stat = -1;
			goto error;
		}
	}
		
	/* Reset the board, via dt2801_reset(),
	   which will call the dt2801_xxx_reset() routines. */
	for (boardPtr = boardlist; boardPtr < boardlist+nboards; boardPtr++) {
		if (dt2801_reset(boardPtr) != 0) {
			printk("dt2801_init: reset not ok.\n");
			goto error;
		}
	}
	
	printk("dt2801: major(%d).\n", major);
        if (nboards == 0) {
		printk("dt2801: NO boards.\n");
	} else {
		printk("dt2801: #of boards: %d.\n", nboards);
		for (boardPtr = boardlist; boardPtr < boardlist + nboards;
		     boardPtr++) {
			printk("dt2801:(%d) %-15s fake(%d), io(0x%lx), dma(%d)\n",
			       boardPtr->boardid,
			       boardPtr->typename,
			       boardPtr->fake,
			       boardPtr->baseaddr,
			       boardPtr->dmachannel);
			dt2801_adc_info(boardPtr);
			dt2801_dac_info(boardPtr);
			dt2801_dio_info(boardPtr);
		}
	}

	/* A module consisting of more than one object-file will
	   export all symbols (even when declared static). So we
	   explicitly remove the symbols from the table.
	   c.f. /usr/doc/modules/HOWTO-modularize. */
	register_symtab(0);

	/* stat = inb_p(dt_status(boardPtr)); */
	/* printk("dt2801: finally: status = 0x%0x\n", stat); */

	return 0;

error:

	printk("dt2801: error during init, cleaning up.\n");
	for (boardPtr = boardlist; boardPtr < boardlist+nboards; boardPtr++) {
		dt2801_adc_cleanup(boardPtr);
		dt2801_dac_cleanup(boardPtr);
		dt2801_dio_cleanup(boardPtr);
	}

fail3:
	unregister_chrdev(major, "dt2801");

fail_major:
	for (boardPtr = boardlist; boardPtr < boardlist+nboards; boardPtr++) {
		if (boardPtr->baseaddr != 0) {
			release_region(boardPtr->baseaddr, DT2801_IOSIZE);
		}
	}

	for (boardPtr = boardlist; boardPtr < boardlist+nboards; boardPtr++) {
		if (boardPtr->dmachannel != -1) {
			free_dma(boardPtr->dmachannel);
		}
	}

	kfree(boardlist);

        /* If I forgot to set 'stat' in case of an error. Force
	   unloading the module */
        if (stat == 0) { stat = -1; }
	return stat;
}

void cleanup_module(void)
{
        Dt2801_t* boardPtr;

	for (boardPtr = boardlist; boardPtr < boardlist+nboards; boardPtr++) {
		dt2801_adc_cleanup(boardPtr);
		dt2801_dac_cleanup(boardPtr);
		dt2801_dio_cleanup(boardPtr);

		if (boardPtr->dmachannel != -1) {
			free_dma(boardPtr->dmachannel);
		}
		release_region(boardPtr->baseaddr, DT2801_IOSIZE);
	}

	if (unregister_chrdev(major, "dt2801") != 0) {
		printk("dt2801: cleanup_module failed\n");
	} else {
		printk("dt2801: cleanup succeeded\n");
	}

	kfree(boardlist);
}

#endif
