/*
   modules/dt282x.c
   hardware driver for Data Translation DT2821 series

   COMEDI - Linux Control and Measurement Device Interface
   Copyright (C) 1997-8 David A. Schleef <ds@stm.lbl.gov>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 */


#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/timex.h>
#include <linux/timer.h>
#include <asm/io.h>
#include <asm/dma.h>
#include <comedi_module.h>

#define DT_DMA 0

#define DT2821_TIMEOUT		100	/* 500 us */
#define DT2821_SIZE 0x10

/*
 *    Registers in the DT282x
 */

#define DT2821_ADCSR	0x00	/* A/D Control/Status             */
#define DT2821_CHANCSR	0x02	/* Channel Control/Status */
#define DT2821_ADDAT	0x04	/* A/D data                       */
#define DT2821_DACSR	0x06	/* D/A Control/Status             */
#define DT2821_DADAT	0x08	/* D/A data                       */
#define DT2821_DIODAT	0x0a	/* digital data                   */
#define DT2821_SUPCSR	0x0c	/* Supervisor Control/Status      */
#define DT2821_TMRCTR	0x0e	/* Timer/Counter          */

/*
 *  At power up, some registers are in a well-known state.  The
 *  masks and values are as follows:
 */

#define DT2821_ADCSR_MASK 0xfff0
#define DT2821_ADCSR_VAL 0x7c00

#define DT2821_CHANCSR_MASK 0xfff0
#define DT2821_CHANCSR_VAL 0x70f0

#define DT2821_DACSR_MASK 0xffff
#define DT2821_DACSR_VAL 0x7c9c

#define DT2821_SUPCSR_MASK 0xfcff
#define DT2821_SUPCSR_VAL 0x0000

#define DT2821_TMRCTR_MASK 0xff00
#define DT2821_TMRCTR_VAL 0xf000

/*
 *    Bit fields of each register
 */

/* ADCSR */

#define DT2821_ADERR	0x8000	/* (R)   1 for A/D error  */
#define DT2821_ADCLK	0x0200	/* (R/W) A/D clock enable */
		/*      0x7c00           read as 1's            */
#define DT2821_MUXBUSY	0x0100	/* (R)   multiplexer busy */
#define DT2821_ADDONE	0x0080	/* (R)   A/D done         */
#define DT2821_IADDONE	0x0040	/* (R/W) interrupt on A/D done    */
		/*      0x0030           gain select            */
		/*      0x000f           channel select         */

/* CHANCSR */

#define DT2821_LLE	0x8000	/* (R/W) Load List Enable */
		/*      0x7000           read as 1's            */
		/*      0x0f00     (R)   present address        */
		/*      0x00f0           read as 1's            */
		/*      0x000f     (R)   number of entries - 1  */

/* DACSR */

#define DT2821_DAERR	0x8000	/* (R)   D/A error                */
#define DT2821_YSEL	0x0200	/* (R/W) DAC 1 select             */
#define DT2821_SSEL	0x0100	/* (R/W) single channel select    */
#define DT2821_DACRDY	0x0080	/* (R)   DAC ready                */
#define DT2821_IDARDY	0x0040	/* (R/W) interrupt on DAC ready   */
#define DT2821_DACLK	0x0020	/* (R/W) D/A clock enable */
#define DT2821_HBOE	0x0002	/* (R/W) DIO high byte output enable      */
#define DT2821_LBOE	0x0001	/* (R/W) DIO low byte output enable       */

/* SUPCSR */

#define DT2821_DMAD	0x8000	/* (R)   DMA done                 */
#define DT2821_ERRINTEN	0x4000	/* (R/W) interrupt on error               */
#define DT2821_CLRDMADNE 0x2000	/* (W)   clear DMA done                   */
#define DT2821_DDMA	0x1000	/* (R/W) dual DMA                 */
#define DT2821_DS1	0x0800	/* (R/W) DMA select 1                     */
#define DT2821_DS0	0x0400	/* (R/W) DMA select 0                     */
#define DT2821_BUFFB	0x0200	/* (R/W) buffer B selected                */
#define DT2821_SCDN	0x0100	/* (R)   scan done                        */
#define DT2821_DACON	0x0080	/* (W)   DAC single conversion            */
#define DT2821_ADCINIT	0x0040	/* (W)   A/D initialize                   */
#define DT2821_DACINIT	0x0020	/* (W)   D/A initialize                   */
#define DT2821_PRLD	0x0010	/* (W)   preload multiplexer              */
#define DT2821_STRIG	0x0008	/* (W)   software trigger         */
#define DT2821_XTRIG	0x0004	/* (R/W) external trigger enable  */
#define DT2821_XCLK	0x0002	/* (R/W) external clock enable            */
#define DT2821_BDINIT	0x0001	/* (W)   initialize board         */

/*
--BEGIN-RANGE-DEFS--
RANGE_dt282x_ai_lo_bipolar
        -10     10
        -5      5
        -2.5    2.5
        -1.25   1.25
RANGE_dt282x_ai_lo_bipolar
        0       10
        0       5
        0       2.5
        0       1.25
RANGE_dt282x_ai_5_bipolar
       -5      5
       -2.5    2.5
       -1.25   1.25
       -0.625  0.625
RANGE_dt282x_ai_5_unipolar
        0       5
        0       2.5
        0       1.25
        0       0.625
RANGE_dt282x_ai_hi_bipolar
        -10     10
        -1      1
        -0.1    0.1
        -0.02   0.02
RANGE_dt282x_ai_hi_unipolar
        0       10
        0       1
        0       0.1
        0       0.02
---END-RANGE-DEFS---
*/


/*
 *    map of boardnames and single ended/differential jumper
 *      to boardtype below
 */
typedef struct {
	char *name;
	int boardcode_se;
	int boardcode_di;
	int t0, t1, t2;
} boardname_t;


static boardname_t boardnames[] =
{
	{"dt2821", 0, 1, RANGE_dt282x_ai_lo_unipolar, RANGE_dt282x_ai_lo_bipolar, 0},
	{"dt2821-f-16se", 0, 0, RANGE_dt282x_ai_lo_unipolar, RANGE_dt282x_ai_lo_bipolar, 0},
	{"dt2821-f-8di", 1, 1, RANGE_dt282x_ai_lo_unipolar, RANGE_dt282x_ai_lo_bipolar, 0},
	{"dt2821-g-16se", 0, 0, RANGE_dt282x_ai_lo_unipolar, RANGE_dt282x_ai_lo_bipolar, RANGE_dt282x_ai_5_bipolar},
	{"dt2821-g-8di", 1, 1, RANGE_dt282x_ai_lo_unipolar, RANGE_dt282x_ai_lo_bipolar, RANGE_dt282x_ai_5_bipolar},
	{"dt2823", 2, 2, (RANGE_bipolar10), RANGE_bipolar10, 0},
	{"dt2824-pgh", 3, 4, RANGE_dt282x_ai_lo_unipolar, RANGE_dt282x_ai_lo_bipolar, 0},
	{"dt2824-pgl", 3, 4, RANGE_dt282x_ai_hi_unipolar, RANGE_dt282x_ai_hi_bipolar, 0},
	{"dt2825", 0, 1, RANGE_dt282x_ai_hi_unipolar, RANGE_dt282x_ai_hi_bipolar, 0},
	{"dt2827", 5, 5, (RANGE_bipolar10), RANGE_bipolar10, 0},
	{"dt2828", 6, 6, RANGE_unipolar10, RANGE_bipolar10, 0}
};
static int n_boardnames = 11;

/*
 *    These boardtypes represent the different possible combinations
 *      of boards that exist.  This includes the A/D input jumper
 *      determining if the inputs are single ended or differential.
 *      Other jumpers are not important.
 */
/* beware, the macros from hell */
#define _12BIT	12, 0xfff, 0x800
#define _16BIT	16, 0xffff, 0x8000
typedef struct {
	int adbits;
	int admask;
	int adsignbit;
	int adgains;
	int adchans;
	int adref;
	int dabits;
	int damask;
	int dasignbit;
	int dachans;
} boardtype_t;
static boardtype_t boardtypes[] =
{
	{_12BIT, 4, 16, AREF_COMMON, _12BIT, 2},	/* 2821, 2825, single ended */
	{_12BIT, 4, 8, AREF_DIFF, _12BIT, 2},	/* 2821, 2825, differential */
	{_16BIT, 1, 4, AREF_DIFF, _16BIT, 2},	/* 2823 */
	{_12BIT, 4, 16, AREF_COMMON, 0, 0, 0, 0},	/* 2824, single ended */
	{_12BIT, 4, 8, AREF_DIFF, 0, 0, 0, 0},	/* 2824, differential */
	{_16BIT, 1, 4, AREF_DIFF, _12BIT, 2},	/* 2827 */
	{_12BIT, 1, 4, AREF_COMMON, _12BIT, 2},	/* 2828 */
};

typedef struct {
	boardtype_t *board;	/* type of board we have        */

	int ad_2scomp;		/* we have 2's comp jumper set  */
	int da0_2scomp;		/* same, for DAC0               */
	int da1_2scomp;		/* same, for DAC1               */

	int dacsr;		/* software copies of registers */
	int adcsr;
	int supcsr;

	int ntrig;
	int curadchan;

	short *dmaptr1, *dmaptr2;	/* pointers to DMA buffers      */
	int dma1, dma2;		/* DMA channels                 */
	int dma1_size, dma2_size;	/* size of DMA transfer         */
	int dma_maxsize;	/* max size of DMA transfer     */
	int usedma;		/* driver uses DMA              */
	int current_dma_chan;
} dt282x_private;

#define devpriv ((dt282x_private *)dev->private)
#define boardtype (*devpriv->board)

/*
 *    Some useless abstractions
 */
#define chan_to_DAC(a)	((a)&1)
#define update_dacsr(a)	outw(devpriv->dacsr|(a),dev->iobase+DT2821_DACSR)
#define update_adcsr(a)	outw(devpriv->adcsr|(a),dev->iobase+DT2821_ADCSR)
#define mux_busy() (inw(dev->iobase+DT2821_ADCSR)&DT2821_MUXBUSY)
#define ad_done() (inw(dev->iobase+DT2821_ADCSR)&DT2821_ADDONE)
#define update_supcsr(a)	outw(devpriv->supcsr|(a),dev->iobase+DT2821_SUPCSR)

/*
 *    danger! macro abuse... a is the expression to wait on, and b is
 *      the statement(s) to execute if it doesn't happen.
 */
#define wait_for(a,b)	 				\
	do{						\
		int _i;					\
		for(_i=0;_i<DT2821_TIMEOUT;_i++){	\
			if(a){_i=0;break;}		\
			udelay(5);			\
		}					\
		if(_i){b}				\
	}while(0)


comedi_devinit dt282x_init;
static int dt282x_rem(comedi_device * dev);
static void free_resources(comedi_device *dev);


#if DT_DMA
/*
 *    We don't really care here if we're doing dual DMA or not,
 *      since we just restart the DMA controller.
 */
static void dt282x_dma_interrupt(comedi_device * dev)
{
	int i, n, dma_chan;
	comedi_sample pt;

	pt.chan = devpriv->curadchan;
	pt.job = dev->busy[0];

	update_supcsr(DT2821_CLRDMADNE);

	if (!devpriv->current_dma_chan) {
		printk("intr dma 0\n");
		/* channel that finished was A */
		disable_dma(devpriv->dma1);
		for (i = 0; i < devpriv->dma1_size; i++) {
			pt.data = devpriv->dmaptr1[i];
			report_sample(dev, &pt);
		}
	} else {
		printk("intr dma 1\n");
		/* channel that finished was B */
		disable_dma(devpriv->dma2);
		for (i = 0; i < devpriv->dma2_size; i++) {
			pt.data = devpriv->dmaptr2[i];
			report_sample(dev, &pt);
		}
	}
	if (!devpriv->ntrig) {
		/* we're done.  let's go home. */

		dev->busy[0] = 0;
		comedi_done(dev,dev->subdevices+0);

		return;
	}
	/* restart the channel */
	n = devpriv->ntrig;
	if (n >= devpriv->dma_maxsize) {
		n = devpriv->dma_maxsize;
	} else {
		/* clear the dual dma flag, making this the last dma segment */
		devpriv->supcsr &= ~(DT2821_DDMA);
		update_supcsr(0);
	}
	devpriv->ntrig -= n;

	if (!devpriv->current_dma_chan) {
		/* channel A */
		devpriv->dma1_size = n;
		dma_chan = devpriv->dma1;
/* since dma functions are all inlined, two copies are better than one */
		{
			long flags;
			save_flags(flags);
			cli();
			clear_dma_ff(5);	/* actual channel doesn't matter */
			set_dma_count(dma_chan, n << 1);
			enable_dma(dma_chan);
			restore_flags(flags);
		}
		devpriv->current_dma_chan = 1;
	} else {
		/* channel B */
		devpriv->dma2_size = n;
		dma_chan = devpriv->dma2;
		{
			long flags;
			save_flags(flags);
			cli();
			clear_dma_ff(5);	/* actual channel doesn't matter */
			set_dma_count(dma_chan, n << 1);
			enable_dma(dma_chan);
			restore_flags(flags);
		}
		devpriv->current_dma_chan = 0;
	}

	return;
}
#endif

#if 0
/*
 *    This function is incomplete
 */
static int setup_dma_chan(comedi_device * dev, int chan, int count)
{
	int n;
	long flags;

	n = count;
	if (n >= dev->dma_maxsize)
		n = devpriv->dma_maxsize;
	dev->ntrig -= n;
	/* ... */
}
#endif

static void dt282x_interrupt(int irq, void *d, struct pt_regs *regs)
{
	comedi_device *dev = d;
	comedi_subdevice *s = dev->subdevices+0;
	unsigned int supcsr, adcsr, dacsr;
	sampl_t data;

	/*
	   XXX these should be rearranged so that the most probable
	   case is near the top.
	 */
#if DT_DMA
	if ((supcsr = inw(dev->iobase + DT2821_SUPCSR)) & DT2821_DMAD) {
		dt282x_dma_interrupt(dev);
		return;
	}
#else
	supcsr=0;	/* annoying warnings... */
#endif
	if ((adcsr = inw(dev->iobase + DT2821_ADCSR)) & DT2821_ADERR) {
		comedi_error(dev, "A/D error");
		return;
	}
	if ((dacsr = inw(dev->iobase + DT2821_DACSR)) & DT2821_DAERR) {
		comedi_error(dev, "D/A error");
		return;
	}
	if (adcsr & DT2821_ADDONE) {
		data = (sampl_t) inw(dev->iobase + DT2821_ADDAT);

		if (--devpriv->ntrig) {
			update_supcsr(DT2821_STRIG);
		} else {
			comedi_done(dev,s);
		}
		return;
	}
}


static void dt282x_load_changain(comedi_device * dev, int n, unsigned int *chanlist)
{
	unsigned int i;
	unsigned int chan, range;

	outw(DT2821_LLE | (n - 1), dev->iobase + DT2821_CHANCSR);
	for (i = 0; i < n; i++) {
		chan = CR_CHAN(chanlist[i]);
		range = CR_RANGE(chanlist[i]);
		update_adcsr((range << 4) | (chan));
	}
	outw(n - 1, dev->iobase + DT2821_CHANCSR);
}


/*
 *    Performs a single A/D conversion.
 *      - Put channel/gain into channel-gain list
 *      - preload multiplexer
 *      - trigger conversion and wait for it to finish
 */
static int dt282x_ai_mode0(comedi_device * dev, comedi_subdevice * s, comedi_trig * it)
{
	devpriv->adcsr = DT2821_ADCLK;
	update_adcsr(0);

	dt282x_load_changain(dev, 1, it->chanlist);

	update_supcsr(DT2821_PRLD);
	wait_for(!mux_busy(),
		 comedi_error(dev, "timeout\n");
		 return -ETIME;
	    );

	update_supcsr(DT2821_STRIG);
	wait_for(ad_done(),
		 comedi_error(dev, "timeout\n");
		 return -ETIME;
	    );

	it->data[0] = inw(dev->iobase + DT2821_ADDAT) & boardtype.admask;
	if (devpriv->ad_2scomp)
		it->data[0] ^= (1 << (boardtype.adbits - 1));

	return 0;
}

static int dt282x_ai_mode1(comedi_device * dev, comedi_subdevice * s, comedi_trig * it)
{
	if (DT_DMA && !devpriv->usedma) {
		dt282x_load_changain(dev,it->n_chan,it->chanlist);

		outw(it->trigvar, dev->iobase + DT2821_TMRCTR);

		devpriv->adcsr = DT2821_ADCLK | DT2821_IADDONE;
		update_adcsr(0);

		update_supcsr(DT2821_PRLD);
		wait_for(!mux_busy(),
			 comedi_error(dev, "timeout\n");
			 return -ETIME;
		    );
		update_supcsr(DT2821_STRIG);

		return 0;
	} else {
#if DT_DMA
		int n;

		if (it->n > 2048)
			return -EINVAL;
		outw(it->trigvar, dev->iobase + DT2821_TMRCTR);

		devpriv->supcsr = DT2821_ERRINTEN | DT2821_DS0;
		update_supcsr(DT2821_CLRDMADNE | DT2821_BUFFB | DT2821_ADCINIT);
		devpriv->adcsr = 0;

		n = devpriv->ntrig;
		if (n > devpriv->dma_maxsize) {
			n = devpriv->dma_maxsize;
			devpriv->supcsr |= DT2821_DDMA;
			update_supcsr(0);
		} else {
			/* clear the dual dma flag, making this the only dma segment */
			devpriv->supcsr &= ~(DT2821_DDMA);
			update_supcsr(0);
		}
		devpriv->ntrig -= n;
		devpriv->dma1_size = n;
/* DMA junk */
		{
			long flags;
			save_flags(flags);
			cli();
			clear_dma_ff(5);	/* actual channel doesn't matter */

			set_dma_mode(devpriv->dma1, DMA_MODE_READ);
			set_dma_addr(devpriv->dma1, virt_to_bus(devpriv->dmaptr1));
			set_dma_count(devpriv->dma1, n << 1);

			enable_dma(devpriv->dma1);
			restore_flags(flags);
		}
		if (devpriv->ntrig > 0) {
			n = devpriv->ntrig;
			if (n > devpriv->dma_maxsize)
				n = devpriv->dma_maxsize;
			devpriv->ntrig -= n;
			devpriv->dma2_size = n;
			devpriv->current_dma_chan = 0;
/* DMA junk */
			{
				long flags;
				save_flags(flags);
				cli();
				clear_dma_ff(5);	/* actual channel doesn't matter */

				set_dma_mode(devpriv->dma2, DMA_MODE_READ);
				set_dma_addr(devpriv->dma2, virt_to_bus(devpriv->dmaptr2));
				set_dma_count(devpriv->dma2, n << 1);

				enable_dma(devpriv->dma2);
				restore_flags(flags);
			}
		}
		devpriv->adcsr = DT2821_ADCLK | DT2821_IADDONE;
		update_adcsr(0);

		dt282x_load_changain(dev,it->n_chan,it->chanlist);

		devpriv->adcsr = DT2821_ADCLK | DT2821_IADDONE;
		update_adcsr(0);

		update_supcsr(DT2821_PRLD);
		wait_for(!mux_busy(),
			 comedi_error(dev, "timeout\n");
			 return -ETIME;
		    );
		update_supcsr(DT2821_STRIG);

		return 0;
#else
		return -EINVAL;
#endif
	}
}


static int dt282x_ai_mode2(comedi_device * dev, comedi_subdevice * s, comedi_trig * it)
{
	if (!dev->irq)
		return -EINVAL;

	devpriv->supcsr = DT2821_ERRINTEN | DT2821_DDMA | DT2821_DS0;
	update_supcsr(DT2821_CLRDMADNE | DT2821_BUFFB | DT2821_ADCINIT);
	devpriv->adcsr = 0;

	devpriv->adcsr = DT2821_ADCLK | DT2821_IADDONE;
	update_adcsr(0);

	dt282x_load_changain(dev,it->n_chan,it->chanlist);

	outw(it->trigvar1, dev->iobase + DT2821_TMRCTR);
	update_supcsr(DT2821_PRLD);
	wait_for(!mux_busy(),
		 comedi_error(dev, "timeout\n");
		 return -ETIME;
	    );
	update_supcsr(DT2821_STRIG);

	return 0;
}

/*
 *    Analog output routine.  Selects single channel conversion,
 *      selects correct channel, converts from 2's compliment to
 *      offset binary if necessary, loads the data into the DAC
 *      data register, and performs the conversion.
 */
static int dt282x_ao(comedi_device * dev, comedi_subdevice * s, comedi_trig * it)
{
	sampl_t data;
	unsigned int chan;

	data = it->data[0];
	chan = CR_CHAN(it->chanlist[0]);

	devpriv->dacsr |= DT2821_SSEL;

	if (chan) {
		/* select channel */
		devpriv->dacsr |= DT2821_YSEL;
		if (devpriv->da0_2scomp)
			data ^= boardtype.dasignbit;
	} else {
		devpriv->dacsr &= ~DT2821_YSEL;
		if (devpriv->da1_2scomp)
			data ^= boardtype.dasignbit;
	}

	update_dacsr(0);

	outw(data, dev->iobase + DT2821_DADAT);

	update_supcsr(DT2821_DACON);

	return 0;
}

static int dt282x_dio(comedi_device * dev, comedi_subdevice * s, comedi_trig * it)
{
#if 0
	switch (it->io) {
	case COMEDI_IO_INPUT:
		it->data = inw(dev->iobase + DT2821_DIODAT);

		return 0;
	case COMEDI_IO_OUTPUT:
		outw(it->data, dev->iobase + DT2821_DIODAT);
		return 0;
	default:
		return -EINVAL;
	}
#else
	return -EINVAL;
#endif
}

#if 0
static int dt282x_dsp(comedi_device * dev, comedi_param * it)
{
	switch (it->pnum) {
	case COMEDI_IOBITS:
		switch (it->pval & 0xff) {
		case 0xff & COMEDI_IOBITS_INPUT:
			devpriv->dacsr &= ~DT2821_LBOE;
			break;
		case 0xff & COMEDI_IOBITS_OUTPUT:
			devpriv->dacsr |= DT2821_LBOE;
			break;
		default:
			return -EINVAL;
		}
		switch (it->pval & 0xff00) {
		case 0xff00 & COMEDI_IOBITS_INPUT:
			devpriv->dacsr &= ~DT2821_HBOE;
			break;
		case 0xff00 & COMEDI_IOBITS_OUTPUT:
			devpriv->dacsr |= DT2821_HBOE;
			break;
		default:
			return -EINVAL;
		}
		outw(devpriv->dacsr, dev->iobase + DT2821_DACSR);
		changeparam(&dev->chinfo[it->chan].paramlist, COMEDI_IOBITS, 0,
			    it->pval);
		return 0;
	default:
		return -EINVAL;
	}
}
#endif

/*
   options:
   i/o base
   irq
   1=differential, 0=single ended
   ai 0=unipolar, 1=bipolar
   ao0 0=unipolar, 1=bipolar
   ao1 0=unipolar, 1=bipolar
   dma1
   dma2
 */

int dt282x_init(comedi_device * dev, comedi_devconfig * it)
{
	int i, irqs, irq;
	long flags;
	int board;
	int board_number;
	int ret;
	comedi_subdevice *s;

	board = -1;
	for (board_number = 0; board_number < n_boardnames; board_number++) {
		if (!strcmp(boardnames[board_number].name, it->board_name)) {
			if (it->options[2])
				board = boardnames[board_number].boardcode_di;
			else
				board = boardnames[board_number].boardcode_se;
			dev->board_name = boardnames[board_number].name;
			break;
		}
	}
	if (board < 0)
		return 0;
	dev->driver_name = "dt282x";

	if (it->options[0])
		dev->iobase = it->options[0];
	else
		dev->iobase = 0x240;

	printk("comedi%d: dt282x: 0x%04x\n", dev->minor, dev->iobase);
	if (check_region(dev->iobase, DT2821_SIZE) < 0) {
		comedi_error(dev, "I/O port conflict");
		return -EIO;
	}
	request_region(dev->iobase, DT2821_SIZE, "dt282x");
	dev->iosize = DT2821_SIZE;

	outw(DT2821_BDINIT, dev->iobase + DT2821_SUPCSR);
	i = inw(dev->iobase + DT2821_ADCSR);
#ifdef DEBUG
	printk("fingerprint=%x,%x,%x,%x,%x",
	       inw(dev->iobase + DT2821_ADCSR),
	       inw(dev->iobase + DT2821_CHANCSR),
	       inw(dev->iobase + DT2821_DACSR),
	       inw(dev->iobase + DT2821_SUPCSR),
	       inw(dev->iobase + DT2821_TMRCTR));
#endif

	if (
		   ((inw(dev->iobase + DT2821_ADCSR) & DT2821_ADCSR_MASK)
		    != DT2821_ADCSR_VAL) ||
	       ((inw(dev->iobase + DT2821_CHANCSR) & DT2821_CHANCSR_MASK)
		!= DT2821_CHANCSR_VAL) ||
		   ((inw(dev->iobase + DT2821_DACSR) & DT2821_DACSR_MASK)
		    != DT2821_DACSR_VAL) ||
		 ((inw(dev->iobase + DT2821_SUPCSR) & DT2821_SUPCSR_MASK)
		  != DT2821_SUPCSR_VAL) ||
		 ((inw(dev->iobase + DT2821_TMRCTR) & DT2821_TMRCTR_MASK)
		  != DT2821_TMRCTR_VAL)) {
		comedi_error(dev, "board not found\n");
		ret= -EIO;
		goto cleanup;
	}
	/* should do board test */

	irq = it->options[1];
	if (irq < 0) {
		save_flags(flags);
		sti();
		irqs = probe_irq_on();

		/* trigger interrupt */

		udelay(100);

		irq = probe_irq_off(irqs);
		restore_flags(flags);
		if (0 /* error */ ) {
			comedi_error(dev, "error probing irq (bad)");
		}
	}
	dev->irq = 0;
	if (irq > 0) {
		printk("( irq = %d )\n", irq);
		request_irq(irq, dt282x_interrupt, SA_INTERRUPT, "dt282x", dev);
		dev->irq = irq;
	} else if (irq == 0) {
		printk("(no irq)\n");
	} else {
		printk("(probe returned multiple irqs--bad)\n");
	}

	if((ret=alloc_private(dev,sizeof(dt282x_private)))<0)
		goto cleanup;

	dev->n_subdevices = 3;
	if((ret=alloc_subdevices(dev))<0)
		goto cleanup;

	devpriv->board = boardtypes + board;

	s=dev->subdevices+0;

	/* ai subdevice */
	s->type=COMEDI_SUBD_AI;
	s->subdev_flags=SDF_READABLE;
	s->n_chan=boardtype.adchans;
	s->trig[0]=dt282x_ai_mode0;
	s->trig[1]=dt282x_ai_mode1;
	s->trig[2]=dt282x_ai_mode2;
	s->maxdata=boardtype.admask;
	s->len_chanlist=16;
	switch (it->options[3]) {
	case 0:
		s->range_type = boardnames[board_number].t0;
		break;
	case 1:
		s->range_type = boardnames[board_number].t1;
		break;
	case 2:
		s->range_type = boardnames[board_number].t2;
		break;
	default:
		s->range_type = 0;
	}
	s->timer_type=TIMER_dt282x;

	s++;
	/* ao subsystem */
	s->type=COMEDI_SUBD_AO;
	s->subdev_flags=SDF_WRITEABLE;
	s->n_chan=boardtype.dachans;
	s->trig[0]=dt282x_ao;
	s->maxdata=boardtype.damask;
	s->len_chanlist=1;			/* XXX can do 2 */
	s->range_type = RANGE_bipolar10;	/* XXX wrong */

	s++;
	/* dio subsystem */
	s->type=COMEDI_SUBD_DIO;
	s->subdev_flags=SDF_READABLE|SDF_WRITEABLE;
	s->n_chan=16;
	s->trig[0]=dt282x_dio;
	s->maxdata=1;
	s->range_type = RANGE_digital;

#if DT_DMA
	/*
 	*    DMA is corrupting memory.  disabled.
 	*/
	if(ret=dt282x_grab_dma(dev,it->options[6],it->options[7]))
		goto cleanup;
#endif

	dev->rem = dt282x_rem;

	return 1;
cleanup:
	free_resources(dev);
	return ret;
}


static void free_resources(comedi_device *dev)
{
	if (dev->irq) {
		free_irq(dev->irq, dev);
	}
	if(dev->iobase)
		release_region(dev->iobase, dev->iosize);
	if(dev->private){
		if (devpriv->dma1)
			free_dma(devpriv->dma1);
		if (devpriv->dma2)
			free_dma(devpriv->dma2);
		if (devpriv->dmaptr1)
			free_page((unsigned long) devpriv->dmaptr1);
		if (devpriv->dmaptr2)
			free_page((unsigned long) devpriv->dmaptr2);
		kfree(dev->private);
	}
	if(dev->subdevices)
		kfree(dev->subdevices);
}

static int dt282x_rem(comedi_device * dev)
{
	printk("comedi%d: dt282x: remove\n", dev->minor);

	free_resources(dev);

	return 0;
}


/*
 *    Start of the DMA allocation atrocity.
 *      This section screams for a rewrite.
 *	rewritten.  looks better now.
 */

#if DT_DMA
static int dt282x_grab_dma(comedi_device *dev,int dma1,int dma2)
{
	int ret;

	devpriv->usedma=0;

	if(dma1==dma2 || dma1<5 || dma2<5 || dma1>7 || dma2>7)
		return -EINVAL;

	if(dma2<dma1){
		int i;i=dma1;dma1=dma2;dma2=i;
	}

	ret = request_dma(dma1, "dt282x");
	if (ret)
		return -EBUSY;
	devpriv->dma1=dma1;

	ret = request_dma(dma2, "dt282x");
	if (ret)
		return -EBUSY;
	devpriv->dma2=dma2;

	devpriv->dma_maxsize = PAGE_SIZE >> 1;
	devpriv->dmaptr1 = (void *) get_free_page(GFP_KERNEL | GFP_DMA);
	devpriv->dmaptr2 = (void *) get_free_page(GFP_KERNEL | GFP_DMA);
	if (!devpriv->dmaptr1 || !devpriv->dmaptr2) {
		printk("can't get DMA memory ");
		return -ENOMEM;
	}

	{
	/* for debugging */
	int i;

	for (i = 0; i < devpriv->dma_maxsize; i++) {
		devpriv->dmaptr1[i] = 0xdead;
		devpriv->dmaptr2[i] = 0xdead;
	}
	}

	devpriv->usedma=1;

	return 0;
}
#endif

