/*
    module/module.c
    comedi kernel module

    COMEDI - Linux Control and Measurement Device Interface
    Copyright (C) 1997-8 David A. Schleef <ds@stm.lbl.gov>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#undef DEBUG

#define MODULE

#include <comedi_module.h>

#include <linux/module.h>
#include <linux/errno.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/fcntl.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <asm/io.h>
#ifdef LINUX_V22
#include <asm/uaccess.h>
#endif

comedi_device *comedi_devices;


static int do_devconfig_ioctl(comedi_device *dev,comedi_devconfig *arg,kdev_t minor);
static int do_devinfo_ioctl(comedi_device *dev,comedi_devinfo *arg);
static int do_subdinfo_ioctl(comedi_device *dev,comedi_subdinfo *arg,void *file);
static int do_chaninfo_ioctl(comedi_device *dev,comedi_chaninfo *arg);
static int do_trig_ioctl(comedi_device *dev,void *arg,void *file);
static int do_lock_ioctl(comedi_device *dev,unsigned int arg,void * file);
static int do_unlock_ioctl(comedi_device *dev,unsigned int arg,void * file);
static int do_cancel_ioctl(comedi_device *dev,unsigned int arg,void *file);

static void postconfig(comedi_device *dev);
static void do_become_nonbusy(comedi_device *dev,comedi_subdevice *s);

static int comedi_ioctl(struct inode * inode,struct file * file,unsigned int cmd,unsigned long arg)
{
	kdev_t minor=MINOR(inode->i_rdev);
	comedi_device *dev=comedi_devices+minor;
	
	switch(cmd)
	{
	case COMEDI_DEVCONFIG:
		return do_devconfig_ioctl(dev,(void *)arg,minor);
	case COMEDI_DEVINFO:
		return do_devinfo_ioctl(dev,(void *)arg);
	case COMEDI_SUBDINFO:
		return do_subdinfo_ioctl(dev,(void *)arg,file);
	case COMEDI_CHANINFO:
		return do_chaninfo_ioctl(dev,(void *)arg);
	case COMEDI_RANGEINFO:
		return do_rangeinfo_ioctl(dev,(void *)arg);
	case COMEDI_TRIG:
		return do_trig_ioctl(dev,(void *)arg,file);
	case COMEDI_LOCK:
		return do_lock_ioctl(dev,arg,file);
	case COMEDI_UNLOCK:
		return do_unlock_ioctl(dev,arg,file);
	case COMEDI_CANCEL:
		return do_cancel_ioctl(dev,arg,file);
	default:
		return -EIO;
	}
}


/*
	COMEDI_DEVCONFIG
	device config ioctl
	
	arg:
		pointer to devconfig structure
	
	reads:
		devconfig structure at arg
	
	writes:
		none
*/
static int do_devconfig_ioctl(comedi_device *dev,comedi_devconfig *arg,kdev_t minor)
{
	comedi_devconfig it;
	int ret,i;
	
	if(!suser())
		return -EPERM;
	
	if(copy_from_user(&it,arg,sizeof(comedi_devconfig)))
		return -EFAULT;
	
	if(strnlen(it.board_name,COMEDI_NAMELEN)==COMEDI_NAMELEN)
		return -EINVAL;
	
	if(dev->rem){
		ret=dev->rem(dev);
		if(ret<0)
			return ret;
	}
	memset(dev,0,sizeof(comedi_device));

	if(!it.board_name[0])return 0;
	
	dev->minor=minor;
	for(i=0;i<comedi_maxinits;i++){
		ret=comedi_devinits[i](dev,&it);
		if(ret<0)
			return ret;
		if(ret){
			/* do a little post-config cleanup */
			postconfig(dev);
			return 0;
		}
	}
	
	return -EINVAL;
}


/*
	COMEDI_DEVINFO
	device info ioctl
	
	arg:
		pointer to devinfo structure
	
	reads:
		none
	
	writes:
		devinfo structure
		
*/
static int do_devinfo_ioctl(comedi_device *dev,comedi_devinfo *arg)
{
	comedi_devinfo devinfo;
	
	
	/* fill devinfo structure */
	devinfo.version_code=COMEDI_VERSION_CODE;
	devinfo.n_subdevs=dev->n_subdevices;
	memcpy(devinfo.driver_name,dev->driver_name,COMEDI_NAMELEN);
	memcpy(devinfo.board_name,dev->board_name,COMEDI_NAMELEN);
	memcpy(devinfo.options,dev->options,COMEDI_NDEVCONFOPTS*sizeof(int));
	

	if(copy_to_user(arg,&devinfo,sizeof(comedi_devinfo)))
		return -EFAULT;

	return 0;
}


/*
	COMEDI_SUBDINFO
	subdevice info ioctl
	
	arg:
		pointer to array of subdevice info structures
	
	reads:
		none
	
	writes:
		array of subdevice info structures at arg
		
*/
static int do_subdinfo_ioctl(comedi_device *dev,comedi_subdinfo *arg,void *file)
{
	int ret,i;
	comedi_subdinfo *tmp,*us;
	comedi_subdevice *s;
	

	tmp=kmalloc(dev->n_subdevices*sizeof(comedi_subdinfo),GFP_KERNEL);
	if(!tmp)
		return -ENOMEM;
	
	/* fill subdinfo structs */
	for(i=0;i<dev->n_subdevices;i++){
		s=dev->subdevices+i;
		us=tmp+i;
		
		us->type		= s->type;
		us->n_chan		= s->n_chan;
		us->subd_flags		= s->subdev_flags;
		us->timer_type		= s->timer_type;
		us->len_chanlist	= s->len_chanlist;
		us->maxdata		= s->maxdata;
		us->range_type		= s->range_type;
		
		if(s->busy)
			us->subd_flags |= SDF_BUSY;
		if(s->busy == file)
			us->subd_flags |= SDF_BUSY_OWNER;
		if(s->lock)
			us->subd_flags |= SDF_LOCKED;
		if(s->lock == file)
			us->subd_flags |= SDF_LOCK_OWNER;
		if(s->maxdata_list)
			us->subd_flags |= SDF_MAXDATA;
		if(s->flaglist)
			us->subd_flags |= SDF_FLAGS;
		if(s->range_type_list)
			us->subd_flags |= SDF_RANGETYPE;
		if(s->trig[0])
			us->subd_flags |= SDF_MODE0;
		if(s->trig[1])
			us->subd_flags |= SDF_MODE1;
		if(s->trig[2])
			us->subd_flags |= SDF_MODE2;
		if(s->trig[3])
			us->subd_flags |= SDF_MODE3;
		if(s->trig[4])
			us->subd_flags |= SDF_MODE4;
	}
	
	ret=copy_to_user(arg,tmp,dev->n_subdevices*sizeof(comedi_subdinfo));
	
	kfree(tmp);
	
	return ret?-EFAULT:0;
}


/*
	COMEDI_CHANINFO
	subdevice info ioctl
	
	arg:
		pointer to chaninfo structure
	
	reads:
		chaninfo structure at arg
	
	writes:
		arrays at elements of chaninfo structure
	
*/
static int do_chaninfo_ioctl(comedi_device *dev,comedi_chaninfo *arg)
{
	comedi_subdevice *s;
	comedi_chaninfo it;
	int ret;
	
	if(copy_from_user(&it,arg,sizeof(comedi_chaninfo)))
		return -EFAULT;
	
	if(it.subdev>=dev->n_subdevices)
		return -EINVAL;
	s=dev->subdevices+it.subdev;
	
	if(it.flaglist){
		if(s->subdev_flags & SDF_FLAGS)
			ret=copy_to_user(it.flaglist,s->flaglist,s->n_chan*sizeof(unsigned int));
		else
			ret=clear_user(it.flaglist,s->n_chan*sizeof(unsigned int));
		if(ret)return -EFAULT;
	}
			
	if(it.rangelist){
		if(s->subdev_flags & SDF_FLAGS)
			ret=copy_to_user(it.rangelist,s->range_list,s->n_chan*sizeof(unsigned int));
		else
			ret=clear_user(it.rangelist,s->n_chan*sizeof(unsigned int));
		if(ret)return -EFAULT;
	}
	
	return 0;
}


/*
	COMEDI_TRIG
	trigger ioctl
	
	arg:
		pointer to trig structure
	
	reads:
		trig structure at arg
		channel/range list
	
	writes:
		modified trig structure at arg
		data list

	this function is too complicated
*/
static int do_trig_ioctl(comedi_device *dev,void *arg,void *file)
{
	comedi_trig user_trig;
	comedi_subdevice *s;
	int ret=0,i,bufsz;
	
	if(copy_from_user(&user_trig,arg,sizeof(comedi_trig)))
		return -EFAULT;
	
#if 0
	/* this appears to be the only way to check if we are allowed
	   to write to an area. */
	if(copy_to_user(arg,&user_trig,sizeof(comedi_trig)))
		return -EFAULT;
#endif
	
	if(user_trig.subdev>=dev->n_subdevices)
		return -ENODEV;

	s=dev->subdevices+user_trig.subdev;
	if(s->type==COMEDI_SUBD_UNUSED)
		return -EIO;
	
	/* are we locked? (ioctl lock) */
	if(s->lock && s->lock!=file)
		return -EACCES;

	/* are we busy? */
	if(s->busy)
		return -EBUSY;
	s->busy=file;

	/* make sure channel/gain list isn't too long */
	if(user_trig.n_chan > s->len_chanlist){
		DPRINTK("channel/gain list too long %d > %d\n",user_trig.n_chan,s->len_chanlist);
		ret = -EINVAL;
		goto cleanup;
	}

	
	s->cur_trig=user_trig;
	s->cur_trig.chanlist=NULL;
	s->cur_trig.data=NULL;

	/* load channel/gain list */
	s->cur_trig.chanlist=kmalloc(s->cur_trig.n_chan*sizeof(int),GFP_KERNEL);
	if(!s->cur_trig.chanlist){
		ret = -ENOMEM;
		goto cleanup;
	}
	
	if(copy_from_user(s->cur_trig.chanlist,user_trig.chanlist,s->cur_trig.n_chan*sizeof(int))){
		ret = -EFAULT;
		goto cleanup;
	}
	
	/* make sure each element in channel/gain list is valid */
	if((ret=check_chanlist(s,s->cur_trig.n_chan,s->cur_trig.chanlist))<0)
		goto cleanup;
	
	/* allocate buffer */
	if(s->flags&SDF_LSAMPL){
		bufsz=s->cur_trig.n*s->cur_trig.n_chan*sizeof(lsampl_t);
	}else{
		bufsz=s->cur_trig.n*s->cur_trig.n_chan*sizeof(sampl_t);
	}
	if(bufsz>s->maxbufsz){
		DPRINTK("buffer size too big %d > %d\n",bufsz,s->maxbufsz);
		/* XXX we *should* be able to handle this */
		ret=-EINVAL;
		goto cleanup;
	}
	if(s->buf){
		s->cur_trig.data=s->buf;
	}else{
		if(!(s->cur_trig.data=kmalloc(bufsz,GFP_KERNEL))){
			ret=-ENOMEM;
			goto cleanup;
		}
	}

/* debugging */
memset(s->cur_trig.data,0xef,bufsz);

	s->buf_int_ptr=0;
	s->buf_user_ptr=0;
	s->buf_len=s->cur_trig.n*s->cur_trig.n_chan;
	
#if 0
	if(user_trig.data==NULL){
		/* XXX this *should* indicate that we want to transfer via read/write */
		ret=-EINVAL;
		goto cleanup;
	}
#endif
	if(s->subdev_flags & SDF_WRITEABLE){
		if(copy_from_user(s->cur_trig.data,user_trig.data,bufsz)){
			ret=-EFAULT;
			goto cleanup;
		}
	}

	if(s->cur_trig.mode>=5 || s->trig[s->cur_trig.mode]==NULL){
		DPRINTK("bad mode %d\n",s->cur_trig.mode);
		ret=-EINVAL;
		goto cleanup;
	}

	/* mark as non-RT operation */
	s->cur_trig.flags &= ~TRIG_RT;

	s->subdev_flags|=SDF_RUNNING;

	ret=s->trig[s->cur_trig.mode](dev,s,&s->cur_trig);
	
	if(ret==0) return 0;

	if(ret<0)goto cleanup;

	if(s->flags&SDF_LSAMPL){
		i=ret*sizeof(lsampl_t);
	}else{
		i=ret*sizeof(sampl_t);
	}
	if(i>bufsz){
		printk("comedi: (bug) trig returned too many samples\n");
		i=bufsz;
	}
	if(s->subdev_flags & SDF_READABLE){
		if(copy_to_user(user_trig.data,s->cur_trig.data,i)){
			ret=-EFAULT;
			goto cleanup;
		}
	}
cleanup:

	do_become_nonbusy(dev,s);
	
	return ret;
}


/*
	COMEDI_LOCK
	lock subdevice
	
	arg:
		subdevice number
	
	reads:
		none
	
	writes:
		none

	non-RT linux always controls rtcomedi_lock_semaphore.  If an
	RT-linux process wants the lock, it first checks rtcomedi_lock_semaphore.
	If it is 1, it knows it is pre-empting this function, and fails.
	Obviously, if RT-linux fails to get a lock, it *must* allow
	linux to run, since that is the only way to free the lock.
	
	This function is not SMP compatible.

	necessary locking:
	- ioctl/rt lock  (this type)
	- lock while subdevice busy
	- lock while subdevice being programmed
	
*/

volatile int rtcomedi_lock_semaphore=0;

static int do_lock_ioctl(comedi_device *dev,unsigned int arg,void * file)
{
	int ret=0;
	comedi_subdevice *s;
	
	if(arg>=dev->n_subdevices)
		return -EINVAL;
	s=dev->subdevices+arg;
	
	if(s->busy)
		return -EBUSY;

	rtcomedi_lock_semaphore=1;
	
	if(s->lock && s->lock!=file){
		ret=-EACCES;
	}else{
		s->lock=file;
	}
	
	rtcomedi_lock_semaphore=0;

	return ret;
}


/*
	COMEDI_UNLOCK
	unlock subdevice
	
	arg:
		subdevice number
	
	reads:
		none
	
	writes:
		none

	This function isn't protected by the semaphore, since
	we already own the lock.
*/
static int do_unlock_ioctl(comedi_device *dev,unsigned int arg,void * file)
{
	comedi_subdevice *s;
	
	if(arg>=dev->n_subdevices)
		return -EINVAL;
	s=dev->subdevices+arg;
	
	if(s->busy)
		return -EBUSY;

	if(s->lock && s->lock!=file)
		return -EACCES;
	
	if(s->lock==file)
		s->lock=NULL;

	return 0;
}

static int do_cancel(comedi_device *dev,comedi_subdevice *s);
/*
	COMEDI_CANCEL
	cancel acquisition ioctl
	
	arg:
		subdevice number
	
	reads:
		nothing
	
	writes:
		nothing

*/
static int do_cancel_ioctl(comedi_device *dev,unsigned int arg,void *file)
{
	comedi_subdevice *s;
	
	if(arg>=dev->n_subdevices)
		return -EINVAL;
	s=dev->subdevices+arg;
	
	if(s->lock && s->lock!=file)
		return -EACCES;
	
	if(!s->busy)
		return 0;

	if(s->busy!=file)
		return -EBUSY;

	return do_cancel(dev,s);
}
	
static int do_cancel(comedi_device *dev,comedi_subdevice *s)
{
	int ret=0;

	if((s->subdev_flags&SDF_RUNNING) && s->cancel)
		ret=s->cancel(dev,s);

	do_become_nonbusy(dev,s);

	return ret;
}

#ifdef LINUX_V22
/*
   comedi_mmap_v22

   mmap issues:
   	- mmap has issues with lock and busy
	- mmap has issues with reference counting
	- RT issues?

   unmapping:
   	vm_ops->unmap()
 */
static int comedi_mmap_v22(struct file * file, struct vm_area_struct *vma)
{
	kdev_t minor=minor_of_file(file);
	comedi_device *dev=comedi_devices+minor;
	comedi_subdevice *s;
	int size;

	if(vma->vm_offset >= dev->n_subdevices)
		return -EIO;
	s=dev->subdevices+vma->vm_offset;

	if(!(s->subdev_flags&SDF_MMAP))
		return -EIO;

	if((vma->vm_flags & VM_WRITE) && !(s->subdev_flags & SDF_WRITEABLE))
		return -EINVAL;

	if((vma->vm_flags & VM_READ) && !(s->subdev_flags & SDF_READABLE))
		return -EINVAL;

	/* check if aleady mmaped */

	size = vma->vm_end - vma->vm_start;

	/* do some size checking */

	if(remap_page_range(vma->vm_start, virt_to_phys(s->buf),
		size,vma->vm_page_prot))
		return -EAGAIN;
	
	vma->vm_file=file;
	file->f_count++;

	/* mark subdev as mapped */
	
	/* call subdev about mmap, if necessary */

	memset(s->buf,0,size);

	return 0;
}
#endif


static ssize_t comedi_write_v22(struct file *file,const char *buf,size_t nbytes,loff_t *offset)
{
#if 0
	int retval;
	kdev_t minor=minor_of_file(file);
	comedi_device *dev=comedi_devices+minor;
#endif

	if(!nbytes)return 0;

	/* read data */
	
	return -EIO;
}


static ssize_t comedi_read_v22(struct file * file,char *buf,size_t nbytes,loff_t *offset)
{
	comedi_device *dev;
	comedi_subdevice *s;
	int n,m,count=0,retval=0;
	struct wait_queue wait={current,NULL};
	int sample_size;

	dev=comedi_devices+minor_of_file(file);
	s=dev->subdevices+file->f_pos;

	if(s->flags&SDF_LSAMPL){
		sample_size=sizeof(lsampl_t);
	}else{
		sample_size=sizeof(sampl_t);
	}
	if(nbytes%sample_size)
		nbytes-=nbytes%sample_size;

	if(!nbytes)return 0;

	if(!s->busy)
		return 0;

	if(!s->cur_trig.data || !(s->subdev_flags&SDF_READABLE))
		return -EIO;

	if(s->busy != file)
		return -EACCES;

	add_wait_queue(&dev->wait,&wait);
	while(nbytes>0 && !retval){
		current->state=TASK_INTERRUPTIBLE;

		n=nbytes;

		m=(s->buf_int_ptr-s->buf_user_ptr)*sample_size;
		if(m<n)n=m;

		if(n==0){
			if(file->f_flags&O_NONBLOCK){
				retval=-EAGAIN;
				break;
			}
			if(signal_pending(current)){
				retval=-ERESTARTSYS;
				break;
			}
			if(!(s->subdev_flags&SDF_RUNNING)){
				do_become_nonbusy(dev,s);
				break;
			}
			schedule();
			continue;
		}
		m=copy_to_user(buf,s->cur_trig.data+s->buf_user_ptr,n);
		if(m) retval=-EFAULT;
		n-=m;
		
		count+=n;
		nbytes-=n;
		s->buf_user_ptr+=n/sample_size;

		if(s->buf_user_ptr>=s->buf_len){
			if(s->subdev_flags & SDF_RUNNING)
				printk("ack.  end of buffer reached!\n");
			else
				do_become_nonbusy(dev,s);
		}

		buf+=n;
		break;	/* makes device work like a pipe */
	}
	current->state=TASK_RUNNING;
	remove_wait_queue(&dev->wait,&wait);

	return (count ? count : retval);
}

/*
   This function restores a subdevice to an idle state.
 */
static void do_become_nonbusy(comedi_device *dev,comedi_subdevice *s)
{
#if 0
	printk("becoming non-busy\n");
#endif
	/* we do this because it's useful for the non-standard cases */
	s->subdev_flags &= ~SDF_RUNNING;

	if(s->cur_trig.chanlist){
		kfree(s->cur_trig.chanlist);
		s->cur_trig.chanlist=NULL;
	}

	if(s->cur_trig.data){
		if(s->cur_trig.data!=s->buf)
			kfree(s->cur_trig.data);

		s->cur_trig.data=NULL;
	}

	s->busy=NULL;
}

/* no chance that these will change soon */
#define SEEK_SET 0
#define SEEK_CUR 1
#define SEEK_END 2

static loff_t comedi_lseek_v22(struct file *file,loff_t offset,int origin)
{
	comedi_device *dev;
	loff_t new_offset;
	
	dev=comedi_devices+minor_of_file(file);

	switch(origin){
	case SEEK_SET:
		new_offset = offset;
		break;
	case SEEK_CUR:
		new_offset = file->f_pos + offset;
		break;
	case SEEK_END:
		new_offset = dev->n_subdevices + offset;
		break;
	default:
		return -EINVAL;
	}
	if(new_offset<0 || new_offset >= dev->n_subdevices)
		return -EINVAL;

	return file->f_pos=new_offset;
}

static int comedi_open(struct inode *inode,struct file *file)
{
	kdev_t minor=minor_of_file(file);
	comedi_device *dev;

	if(minor>=COMEDI_NDEVICES)return -ENODEV;

	dev=comedi_devices+minor;
	if(!dev->rem && !suser())
		return -ENODEV;
	MOD_INC_USE_COUNT;
	dev->use_count++;
	
	file->private_data=NULL;

	return 0;
}

static int comedi_close_v22(struct inode *inode,struct file *file)
{
	comedi_device *dev=comedi_devices+minor_of_file(file);
	comedi_subdevice *s;
	int i;

	for(i=0;i<dev->n_subdevices;i++){
		s=dev->subdevices+i;
		
		if(s->busy==file){
			do_cancel(dev,s);
		}
		if(s->lock==file){
			s->lock=NULL;
		}
	}
	
	MOD_DEC_USE_COUNT;
	dev->use_count--;
	
	return 0;
}


/*
	kernel compatibility
*/

#ifdef LINUX_V20

static int comedi_write_v20(struct inode *inode,struct file *file,const char *buf,int nbytes)
{
	return comedi_write_v22(file,buf,nbytes,NULL);
}

static int comedi_read_v20(struct inode *inode,struct file *file,char *buf,int nbytes)
{
	return comedi_read_v22(file,buf,nbytes,NULL);
}

static int comedi_lseek_v20(struct inode * inode,struct file *file,off_t offset,int origin)
{
	return comedi_lseek_v22(file,offset,origin);
}

static void comedi_close_v20(struct inode *inode,struct file *file)
{
	comedi_close_v22(inode,file);
}

#define comedi_ioctl_v20 comedi_ioctl
#define comedi_open_v20 comedi_open

static struct file_operations comedi_fops={
	lseek		: comedi_lseek_v20,
	ioctl		: comedi_ioctl_v20,
	open		: comedi_open_v20,
	release		: comedi_close_v20,
	read		: comedi_read_v20,
	write		: comedi_write_v20,
};

#endif

#ifdef LINUX_V22

#define comedi_ioctl_v22 comedi_ioctl
#define comedi_open_v22 comedi_open

static struct file_operations comedi_fops={
	llseek		: comedi_lseek_v22,
	ioctl		: comedi_ioctl_v22,
	open		: comedi_open_v22,
	release		: comedi_close_v22,
	read		: comedi_read_v22,
	write		: comedi_write_v22,
	mmap		: comedi_mmap_v22,
};
#endif



int init_module(void)
{
	printk("comedi: version " COMEDI_VERSION " - David Schleef <ds@stm.lbl.gov>\n");
	if(register_chrdev(COMEDI_MAJOR,"comedi",&comedi_fops)){
		printk("comedi: unable to get major %d\n",COMEDI_MAJOR);
		return -EIO;
	}
	comedi_devices=(comedi_device *)kmalloc(sizeof(comedi_device)*COMEDI_NDEVICES,GFP_KERNEL);
	if(!comedi_devices)
		return -ENOMEM;
	memset(comedi_devices,0,sizeof(comedi_device)*COMEDI_NDEVICES);
	init_polling();

#ifdef CONFIG_EXPORT
#ifdef LINUX_V20
	register_symtab(&comedi_syms);
#endif
#endif

	/* XXX requires /proc interface */
	comedi_proc_init();
	
	return 0;
}

void cleanup_module(void)
{
	int i;

	if(MOD_IN_USE)
		printk("comedi: module in use -- remove delayed\n");
	comedi_proc_cleanup();
	for(i=0;i<COMEDI_NDEVICES;i++){
		if(comedi_devices[i].rem)
			comedi_devices[i].rem(comedi_devices+i);
	}
	cleanup_polling();
	kfree(comedi_devices);

	unregister_chrdev(COMEDI_MAJOR,"comedi");
}


void comedi_error(comedi_device *dev,const char *s)
{
	printk("comedi%d: %s: %s\n",dev->minor,dev->driver_name,s);
}

void comedi_done(comedi_device *dev,comedi_subdevice *s)
{
	DPRINTK("comedi_done\n");

	if(!(s->cur_trig.flags&TRIG_RT))
		wake_up_interruptible(&dev->wait);
	else if(s->end_of_acquisition)
		s->end_of_acquisition(s->eoa_arg);

	s->subdev_flags &= ~SDF_RUNNING;
}

/*
   this function should be called by your interrupt routine
   at end-of-scan events
 */
void comedi_eos(comedi_device *dev,comedi_subdevice *s)
{
	if(!(s->cur_trig.flags&TRIG_RT)){
		if((s->cur_trig.flags&TRIG_WAKE_EOS))
			wake_up_interruptible(&dev->wait);
	}else if(s->end_of_scan){
		int ret;

		ret=s->end_of_scan(s->eos_arg);
		if(ret){
			/* this is a little hack */
			s->buf_int_ptr=0;
			s->cur_chan=0;
		}
	}
}

static void postconfig(comedi_device *dev)
{
	int i;
	comedi_subdevice *s;

	for(i=0;i<dev->n_subdevices;i++){
		s=dev->subdevices+i;

		if(s->type==COMEDI_SUBD_UNUSED)
			continue;

		if(s->len_chanlist==0)
			s->len_chanlist=1;

		if(!s->maxbufsz)
			s->maxbufsz=16*4096;	/* XXX a little low... */

		if(!s->range_type && !s->range_type_list)
			s->range_type=RANGE_unknown;
	}

}

