/*
 * Device Driver for DataTranslation DT2801
 *
 */

#include <comedi_module.h>
#include <linux/delay.h>
#include <linux/ioport.h>
#include <asm/io.h>


/* Hardware Configuration */
/* ====================== */

#define DT2801_MAX_DMA_SIZE (64 * 1024)

/* Ports */
#define DT2801_IOSIZE 2


/* define's & typedef's */
/* ====================== */

/* Commands */
#define DT_C_RESET       0x0
#define DT_C_CLEAR_ERR   0x1
#define DT_C_READ_ERRREG 0x2
#define DT_C_SET_CLOCK   0x3

#define DT_C_TEST        0xb
#define DT_C_STOP        0xf

#define DT_C_SET_DIGIN   0x4
#define DT_C_SET_DIGOUT  0x5
#define DT_C_READ_DIG    0x6
#define DT_C_WRITE_DIG   0x7

#define DT_C_WRITE_DAIM  0x8
#define DT_C_SET_DA      0x9
#define DT_C_WRITE_DA    0xa

#define DT_C_READ_ADIM   0xc
#define DT_C_SET_AD      0xd
#define DT_C_READ_AD     0xe

/* Command modifiers (only used with read/write), EXTTRIG can be
   used with some other commands.
*/
#define DT_MOD_DMA     (1<<4)
#define DT_MOD_CONT    (1<<5)
#define DT_MOD_EXTCLK  (1<<6)
#define DT_MOD_EXTTRIG (1<<7)

/* Bits in status register */
#define DT_S_DATA_OUT_READY   (1<<0)
#define DT_S_DATA_IN_FULL     (1<<1)
#define DT_S_READY            (1<<2)
#define DT_S_COMMAND          (1<<3)
#define DT_S_COMPOSITE_ERROR  (1<<7)

#define DT2801_TIMEOUT 2

/* registers */
#define DT2801_DATA		0
#define DT2801_STATUS		1
#define DT2801_CMD		1


typedef struct{
	char *name;
	int boardcode;
	int adbits;
	int adrangetype;
	int dabits;
} boardtype_t;

/* Typeid's for the different boards of the DT2801-series
   (taken from the test-software, that comes with the board)
   */
static boardtype_t boardtypes[] =
{
	{"dt2801",0x09,		12,RANGE_unknown,12},
	{"dt2801-a",0x52,	12,RANGE_unknown,12},
	{"dt2801/5716a",0x82,	16,RANGE_unknown,12},
	{"dt2805",0x12,		12,RANGE_unknown,12},
	{"dt2805/5716a",0x92,	16,RANGE_unknown,12},
	{"dt2808",0x20,		12,RANGE_unknown,8},
	{"dt2818",0xa2,		12,RANGE_unknown,12},
	{"dt2809",0xb0,		12,RANGE_unknown,12},
};


typedef struct{
	boardtype_t *board;
	unsigned int dac_range_types[2];
}dt2801_private;
#define devpriv ((dt2801_private *)dev->private)
#define boardtype (*devpriv->board)


static int dt2801_ai_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);
static int dt2801_ao_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);
static int dt2801_dio(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);

/* These are the low-level routines:
   writecommand: write a command to the board
   writedata: write data byte
   readdata: read data byte

   They check the status. If the board is not ready the sleep by
   calling dt2801_wait() and try again. If the board is still not
   ready they return with a value != 0 */


void dt2801_wait(comedi_device *dev)
{
	interruptible_sleep_on_timeout(&dev->wait,DT2801_TIMEOUT);

#if 0
	dt2801_start_timer(boardPtr);
 
	interruptible_sleep_on(&boardPtr->waitq);
	boardPtr->dt_timer_int = 0;
	del_timer(&boardPtr->dt_timer_list);
#endif
}


/* Only checks DataOutReady-flag, not the Ready-flag as it is done
   in the examples of the manual. I don't see why this should be
   necessary. */
static int dt2801_readdata(comedi_device *dev, int* data)
{
	int stat = 0;

	stat = inb_p(dev->iobase+DT2801_STATUS);
	if (stat & DT_S_COMPOSITE_ERROR) {
            return -1;
	}

	if (!(stat&DT_S_DATA_OUT_READY)) {
		dt2801_wait(dev);

		stat = inb_p(dev->iobase+DT2801_STATUS);

		if (!(stat&DT_S_DATA_OUT_READY)) {
			printk("dt2801: readdata timed out. "
			       "DataOutReady not set\n");
			return -1;
		}

	}
	*data = inb_p(dev->iobase+DT2801_DATA);

	return 0;
}

static int dt2801_readdata2(comedi_device *dev, int *data)
{
	int lb, hb;

	if (dt2801_readdata(dev, &lb) != 0) {
		return -1;
	}
	if (dt2801_readdata(dev, &hb) != 0) {
		return -2;
	}

	*data =  (hb<<8)+lb;
	return 0;
}

static int dt2801_writedata(comedi_device *dev, unsigned int data)
{
	int stat = 0;

	stat = inb_p(dev->iobase+DT2801_STATUS);

	if (stat & DT_S_COMPOSITE_ERROR) {
            return -1;
	}
	if (stat & DT_S_DATA_IN_FULL) {
		dt2801_wait(dev);
		stat = inb_p(dev->iobase+DT2801_STATUS);

		if (stat & DT_S_DATA_IN_FULL) {
			printk("dt2801: writedata timed out. "
				"DataInFull not cleared\n");
			return -1;
		}
	}
	outb_p(data & 0xff, dev->iobase+DT2801_DATA);

	return 0;
}

static int dt2801_writedata2(comedi_device *dev, unsigned int data)
{

	if ( dt2801_writedata(dev,  data & 0xff) != 0) {
		return -1;
	}
	if ( dt2801_writedata(dev,  (data >> 8) & 0xff ) != 0) {
		return -1;
	}

	return 0;
}

static int dt2801_writecmd(comedi_device * dev, int command)
{
	int stat;

	stat = inb_p(dev->iobase+DT2801_STATUS);
	if (stat & DT_S_COMPOSITE_ERROR) {
        	printk("dt2801: writecommand : warning composite-error\n");
	}

	if (stat & DT_S_DATA_IN_FULL) {
		dt2801_wait(dev);

		stat = inb_p(dev->iobase+DT2801_STATUS);

		if (stat & DT_S_DATA_IN_FULL) {
			printk("dt2801: writecommand timed out. "
			       "DataInFull not cleared\n");
			return -1;
		}
	}

	stat = inb_p(dev->iobase+DT2801_STATUS);

	if (!(stat&DT_S_READY)) {
		dt2801_wait(dev);

		stat = inb_p(dev->iobase+DT2801_STATUS);

		if (!(stat&DT_S_READY)) {
			printk("dt2801: writecommand timed out."
			       "Ready not set\n");
			return -1;
		}

	}
	outb_p(command, dev->iobase+DT2801_CMD);
	return 0;
}


static int dt2801_do_reset(comedi_device *dev)
{
	int stat, board_type;

	stat = inb_p(dev->iobase+DT2801_STATUS);
	/* printk("dt2801: before reset: status = 0x%0x\n", stat); */
	/* stat != 0 if DataInFull-Flag is cleared */
	stat= ((stat)^DT_S_DATA_IN_FULL) & DT_S_DATA_IN_FULL;
        if (!stat) {
		dt2801_wait(dev);
		stat = inb_p(dev->iobase+DT2801_STATUS);
		stat= ((stat)^DT_S_DATA_IN_FULL) & DT_S_DATA_IN_FULL;
		if (!stat) {
			printk("dt2801: do_reset timed out. "
			       "DataInFull not cleared\n");
			printk("dt2801: status = 0x%0x\n", stat);
		        return -1;
		}
	}

	stat = inb_p(dev->iobase+DT2801_STATUS);
	/* printk("dt2801: rst-1 status = 0x%0x\n", stat); */
	/* stat != 0 if Ready-Flag is set */
	stat = stat & DT_S_READY;
        if (!stat) {
		dt2801_wait(dev);
		stat = inb_p(dev->iobase+DT2801_STATUS);
		stat = stat & DT_S_READY;
		if (!stat) {
			printk("dt2801: do_reset timed out. "
			       "Ready not set\n");
			printk("dt2801: status = 0x%0x\n", stat);
		        return -1;
		}
	}
	outb_p(DT_C_RESET, dev->iobase+DT2801_CMD);

	/* This is basically readdata(), but completes if
	   Ready OR DataOutReady are set. */
	stat = inb_p(dev->iobase+DT2801_STATUS);
	/* printk("dt2801: rst-2 status = 0x%0x\n", stat); */
	stat = stat & (DT_S_DATA_OUT_READY);
        if (!stat) {
		dt2801_wait(dev);
		stat = inb_p(dev->iobase+DT2801_STATUS);
	        /* printk("dt2801: rst-3 status = 0x%0x\n", stat); */
		stat = stat & ( DT_S_DATA_OUT_READY);
		if (!stat) {
			printk("dt2801: do_reset timed out. "
			       "Ready not set\n");
			printk("dt2801: status = 0x%0x\n", stat);
		        return -1;
		}
	}
	dt2801_wait(dev);
	board_type = inb_p(dev->iobase+DT2801_DATA);
	dt2801_wait(dev);

	stat = inb_p(dev->iobase+DT2801_STATUS);
	/* printk("dt2801: after reset: status = 0x%0x\n", stat); */

	return board_type;
}

static int dac_range_lkup(int bip,int v10)
{
	if(bip){
		if(v10)return RANGE_unipolar10;
		return RANGE_unipolar5;
	}else{
		if(v10)return RANGE_bipolar10;
		return RANGE_bipolar5;
	}
}

static int dt2801_rem(comedi_device *dev);
/*
   options:
	[0] - i/o base
	[1] - unused
	[2] - a/d 0=differential, 1=single-ended
	[3] - dac0 unipolar=0, bipolar=1
	[4] - dac0 5 V reference =0, 10 V ref = 1
	[5] - dac1 unipolar=0, bipolar=1
	[6] - dac0 5 V reference =0, 10 V ref = 1
*/

int dt2801_init(comedi_device *dev,comedi_devconfig *it)
{
	comedi_subdevice *s;
	int iobase;
	int ret=0;

	if(strcmp("dt2801",it->board_name))
		return 0;

	iobase=it->options[0];
	if(check_region(iobase,DT2801_IOSIZE)<0){
		comedi_error(dev,"I/O port conflict");
		return -EIO;
	}
	request_region(dev->iobase, DT2801_IOSIZE, "dt2801");
	dev->iobase=iobase;

	/* do some checking */

	devpriv->board = boardtypes+ dt2801_do_reset(dev);

	dev->n_subdevices=4;

	if((ret=alloc_subdevices(dev))<0)
		goto cleanup;

	if((ret=alloc_private(dev,sizeof(dt2801_private)))<0)
		goto cleanup;

	s=dev->subdevices+0;
	/* ai subdevice */
	s->type=COMEDI_SUBD_AI;
	s->subdev_flags=SDF_READABLE;
	if(it->options[2])s->n_chan=16;
	else s->n_chan=8;
	s->maxdata=(1<<boardtype.adbits)-1;
	s->range_type=boardtype.adrangetype;
	s->trig[0]=dt2801_ai_mode0;

	s++;
	/* ao subdevice */
	s->type=COMEDI_SUBD_AO;
	s->subdev_flags=SDF_WRITEABLE;
	s->n_chan=2;
	s->maxdata=(1<<boardtype.dabits)-1;
	s->range_type_list=devpriv->dac_range_types;
	devpriv->dac_range_types[0]=dac_range_lkup(it->options[3],it->options[4]);
	devpriv->dac_range_types[1]=dac_range_lkup(it->options[5],it->options[6]);
	s->trig[0]=dt2801_ao_mode0;

	s++;
	/* 1st digital subdevice */
	s->type=COMEDI_SUBD_DIO;
	s->subdev_flags=SDF_READABLE|SDF_WRITEABLE;
	s->n_chan=8;
	s->maxdata=1;
	s->range_type=RANGE_digital;
	s->trig[0]=dt2801_dio;

	s++;
	/* 2nd digital subdevice */
	s->type=COMEDI_SUBD_DIO;
	s->subdev_flags=SDF_READABLE|SDF_WRITEABLE;
	s->n_chan=8;
	s->maxdata=1;
	s->range_type=RANGE_digital;
	s->trig[0]=dt2801_dio;

	dev->rem=dt2801_rem;

	return 1;
cleanup:
	dt2801_rem(dev);

	return ret;
}

static int dt2801_rem(comedi_device *dev)
{
	if(dev->iobase)
		release_region(dev->iobase,DT2801_IOSIZE);

	return 0;
}

static int dt2801_ai_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
        int data;
        int stat;


        stat = dt2801_writecmd(dev, DT_C_READ_ADIM);
        dt2801_writedata(dev, CR_RANGE(it->chanlist[0]));
        dt2801_writedata(dev, CR_CHAN(it->chanlist[0]));
        stat = dt2801_readdata2(dev, &data);

        if (stat != 0) {
             printk("dt2801: stat = %x\n", stat);
             return -EIO;
        }

	it->data[0]=data;

        return 1;
}

static int dt2801_ao_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	int chan=CR_CHAN(it->chanlist[0]);

	dt2801_writecmd(dev, DT_C_WRITE_DAIM);
	dt2801_writedata(dev, chan);
	dt2801_writedata2(dev, it->data[0]);
	
	return 1;
}

static int dt2801_dio(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	unsigned int bits;
	int which=0;
	
	if(s==dev->subdevices+4)which=1;
	
	if(it->flags & TRIG_CONFIG){
		/* configure */
		if(it->chanlist[it->n_chan-1]){
			s->io_bits=0xff;
			dt2801_writecmd(dev, DT_C_SET_DIGOUT);
		}else{
			s->io_bits=0;
			dt2801_writecmd(dev, DT_C_SET_DIGIN);
		}
		dt2801_writedata(dev, which);
	}else{
		if(s->io_bits){
			do_pack(&s->state,it);
			dt2801_writecmd(dev, DT_C_WRITE_DIG);
			dt2801_writedata(dev, which);
			dt2801_writedata(dev, s->state);
		}else{
			dt2801_writecmd(dev, DT_C_READ_DIG);
			dt2801_writedata(dev, which);
			dt2801_readdata(dev, &bits);
			di_unpack(bits,it);
		}
	}
	return it->n_chan;
}



