/*
    module/ni-E.c
    Hardware driver for NI E series cards

    COMEDI - Linux Control and Measurement Device Interface
    Copyright (C) 1997-9 David A. Schleef <ds@stm.lbl.gov>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/*
	This file needs to be included by another file, e.g.,
	atmio-E.c.

	Interrupt support added by Truxton Fulton <trux@truxton.com>

	References for specifications:
	
	   321747b.pdf  Register Level Programmer Manual (obsolete)
	   321747c.pdf  Register Level Programmer Manual (new)
	   DAQ-STC reference manual

	Other possibly relevant info:
	
	   320517c.pdf  User manual (obsolete)
	   320517f.pdf  User manual (new)
	   320889a.pdf  delete
	   320906c.pdf  maximum signal ratings
	   321066a.pdf  about 16x
	   321791a.pdf  discontinuation of at-mio-16e-10 rev. c
	   321808a.pdf  about at-mio-16e-10 rev P
	   321837a.pdf  discontinuation of at-mio-16de-10 rev d
	   321838a.pdf  about at-mio-16de-10 rev N
	
	ISSUES:

	deal with at-mio-16de-10 revision D to N changes, etc.
	
*/

#include <8255.h>


/* reference: ground, common, differential, other */
static int ni_modebits1[4]={ 0x3000, 0x2000, 0x1000, 0 };
static int ni_modebits2[4]={ 0x3f, 0x3f, 0x37, 0x37 };

static int ni_gainlkup[][16]={
	{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15 },
	{ 1, 2, 4, 7, 9, 10, 12, 15, 0,0,0,0,0,0,0,0 },
	{ 1, 2, 3, 4, 5, 6, 7, 9, 10, 11, 12, 13, 14, 15, 0,0 },
	{ 0, 1, 4, 7, 8, 9, 12, 15, 0, 0, 0, 0, 0, 0, 0, 0 }
};

static int ni_range_lkup[]={
	RANGE_ni_E_ai,
	RANGE_ni_E_ai_limited,
	RANGE_ni_E_ai_limited14,
	RANGE_ni_E_ai_limited_602x
};

/*
--BEGIN-RANGE-DEFS--
RANGE_ni_E_ai
        -10     10
        -5      5
        -2.5    2.5
        -1      1
        -0.5    0.5
        -0.25   0.25
        -0.1    0.1
        -0.05   0.05
        0       20
        0       10
        0       5
        0       2
        0       1
        0       0.5
        0       0.2
        0       0.1
RANGE_ni_E_ai_limited
	-10	10
	-5	5
	-1	1
	-0.1	0.1
	0	10
	0	5
	0	1
	0	0.1
RANGE_ni_E_ai_limited14
        -5      5
        -2.5    2.5
        -1      1
        -0.5    0.5
        -0.25   0.25
        -0.1    0.1
        -0.05   0.05
        0       10
        0       5
        0       2
        0       1
        0       0.5
        0       0.2
        0       0.1
RANGE_ni_E_ai_limited_602x
        -10     10
        -5      5
        -0.5    0.5
        -0.05   0.05
        0       20
        0       10
        0       1
        0       0.1
RANGE_ni_E_ao
        -10     10
        0       10
RANGE_ni_E_ao_ext
        -10     10
        0       10
        -1      1       ext
        0       1       ext
---END-RANGE-DEFS---
*/



static int ni_dio(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);
static int ni_rem(comedi_device *dev);
static int ni_read_eeprom(comedi_device *dev,int addr);

static int ni_eeprom(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);
static int ni_calib(comedi_device *dev,comedi_subdevice *s,comedi_trig *it);


static void ni_handle_fifo_half_full(comedi_device *dev);
static void ni_handle_fifo_dregs(comedi_device *dev);

static int ni_ao_fifo_half_empty(comedi_device *dev,comedi_subdevice *s);

static int ni_8255_callback(int dir,int port,int data,void *arg);

#undef TIMEMARK
#undef DEBUG

#ifdef TIMEMARK
#include <linux/time.h>
#endif

static void do_interrupt(comedi_device *dev);

#define RTirqmask_AI 1

#ifdef CONFIG_COMEDI_RT
#ifdef RTL_V1
static void ni_E_rt_interrupt(void);
#else
static unsigned int ni_E_rt_interrupt(unsigned int irq,struct pt_regs *regs);
#endif
static comedi_device *ni_rt_interrupt_dev;

static int grab_rt_irq(comedi_device *dev,int mask)
{
	int ret;

	if(!devpriv->rt_irq){
		ret=rtl_request_global_irq(dev->irq,ni_E_rt_interrupt);
		if(ret<0){
			rt_printk("atmio-E: RT irq not available\n");
			return -EINVAL;
		}
		devpriv->rt_irq=dev->irq;
		devpriv->irqmask=mask;
	}else{
		devpriv->irqmask|=mask;
	}
	ni_rt_interrupt_dev=dev;

	return 0;
}

static int ungrab_rt_irq(comedi_device *dev,int mask)
{
	if(!(devpriv->irqmask&mask))return 0;

	devpriv->irqmask&=~mask;
	if(devpriv->irqmask)return 0;

	if(devpriv->rt_irq){
		rtl_free_global_irq(devpriv->rt_irq);
		devpriv->rt_irq=0;
	}else{
		rt_printk("ni-E: RT irq already free\n");
	}
	return 0;
}

#if LINUX_VERSION_CODE < 0x020200
static void ni_E_rt_interrupt(void)
{
	/* XXX this is terrible, but it's a limitation of RTLinux */
	do_interrupt(ni_rt_interrupt_dev);
}
#else
static unsigned int ni_E_rt_interrupt(unsigned int irq,struct pt_regs *regs)
{
	/* XXX this is terrible, but it's a limitation of RTLinux */
	do_interrupt(ni_rt_interrupt_dev);

	return 0;
}
#endif

#else /* !CONFIG_COMEDI_RT */
#define grab_rt_irq(a,b)
#define ungrab_rt_irq(a,b)
#endif

static void ni_E_interrupt(int irq,void *d,struct pt_regs * regs)
{
	do_interrupt(d);
}

static void do_interrupt(comedi_device *dev)
{
#ifdef TIMEMARK
	struct timeval tv;
#endif
	comedi_subdevice *s=dev->subdevices;
	int status;
	unsigned short b_status;
	int ack=0;
	int wsave;

/*
    If you want to use windowed registers in an interrupt, it is
    important that you restore the window address register.  If
    you change certain modes, e.g., AI_Configuration_Start/End,
    you need to set up software flags for non-interrupt routines.
*/
	wsave=win_save();
	
	b_status=ni_readw(AO_Status_1);
#if 0
printk("B status=0x%04x\n",b_status);
#endif
	status=ni_readw(AI_Status_1);
	if(status&(AI_Overrun_St|AI_Overflow_St)){
		rt_printk("ni_E: overrun/overflow status=0x%04x\n",status);
		comedi_done(dev,s);
		ungrab_rt_irq(dev,RTirqmask_AI);
		return;
	}

#ifdef TIMEMARK
do_gettimeofday(&tv);
rt_printk("ni-E %d.%06d status=0x%04x\n",tv.tv_sec,tv.tv_usec,status);
#endif

#if 0
	/* this is wrong, since we do AO now  */
	if(!dev->subdevices->cur_trig.data){
		rt_printk("aiee! cur_trig.data got zapped!\n");
		comedi_done(dev,s);
		ungrab_rt_irq(dev,RTirqmask_AI);
		return;
	}
#endif

	if(status&AI_SC_TC_St){
#ifdef DEBUG
rt_printk("ni-E: SC_TC interrupt\n");
#endif
		ni_handle_fifo_dregs(dev);
		win_out(0x0000,Interrupt_A_Enable_Register);
		comedi_done(dev,s);
		ungrab_rt_irq(dev,RTirqmask_AI);

		ack|=AI_SC_TC_Interrupt_Ack;
	}
	if(status&AI_FIFO_Half_Full_St){
		ni_handle_fifo_half_full(dev);
	}
	if(!(status&AI_FIFO_Empty_St)
	   && ((dev->subdevices->cur_trig.flags&TRIG_WAKE_EOS)
	      || (dev->subdevices->cb_mask&COMEDI_CB_EOS))){
		ni_handle_fifo_dregs(dev);

		if(s->buf_int_count>=s->cur_chan){
			while(s->buf_int_count>=s->cur_chan)
				s->cur_chan+=s->cur_trig.n_chan*sizeof(sampl_t);
			comedi_eos(dev,dev->subdevices+0);
		}
	}

	if(b_status&AO_Overrun_St){
		printk("ni-E: AO FIFO underrun status=0x%04x status2=0x%04x\n",b_status,ni_readw(AO_Status_2));
	}

	if(b_status&AO_BC_TC_St){
		printk("ni-E: AO BC_TC status=0x%04x status2=0x%04x\n",b_status,ni_readw(AO_Status_2));
	}

	if(b_status&AO_FIFO_Request_St)
		ni_ao_fifo_half_empty(dev,dev->subdevices+1);

	b_status=ni_readw(AO_Status_1);
	if(b_status&Interrupt_B_St){
		if(b_status&AO_FIFO_Request_St){
			printk("AO buffer underrun\n");
		}
		printk("Ack! didn't clear AO interrupt. b_status=0x%04x\n",b_status);
		win_out(0,Interrupt_B_Enable_Register);
	}

	if(ack){
		/* the AI_Configuration_Start is required */

		win_out(AI_Configuration_Start,Joint_Reset_Register);
		ni_writew(ack,Interrupt_A_Ack_Register); /* XXX "_Register" is wrongo... */
		win_out(AI_Configuration_End,Joint_Reset_Register);
	}

	win_restore(wsave);
}

static void ni_ai_fifo_read(comedi_device *dev,comedi_subdevice *s,
		sampl_t *data,int n)
{
	int i;

	for(i=0;i<n;i++){
		data[i]=ni_readw(ADC_FIFO_Data_Register);
	}
}


static void ni_handle_fifo_half_full(comedi_device *dev)
{
	int n,m;
	comedi_subdevice *s=dev->subdevices+0;

	/*
	   if we got a fifo_half_full interrupt, we can transfer fifo/2
	   samples without checking the empty flag.  It doesn't matter if
	   we transfer the rest of the samples, the performance trade-off
	   is minimal (checking empty flag for a few samples vs. having
	   1% more interrupts.)  At really high speeds, it's better to
	   ignore them.

	*/
	
	n=boardtype.ai_fifo_depth/2;

	/* this makes the assumption that the buffer length is
	   greater than the half-fifo depth. */

	if(s->buf_int_ptr+n*sizeof(sampl_t)>=s->cur_trig.data_len){
		m=(s->cur_trig.data_len-s->buf_int_ptr)/sizeof(sampl_t);
		ni_ai_fifo_read(dev,s,((void *)(s->cur_trig.data))+s->buf_int_ptr,m);
		s->buf_int_count+=m*sizeof(sampl_t);
		n-=m;
		s->buf_int_ptr=0;

		comedi_eobuf(dev,s);
	}
	ni_ai_fifo_read(dev,s,((void *)(s->cur_trig.data))+s->buf_int_ptr,n);
	s->buf_int_count+=n*sizeof(sampl_t);
	s->buf_int_ptr+=n*sizeof(sampl_t);

	comedi_bufcheck(dev,s);
}

/*
   Empties the AI fifo
*/
static void ni_handle_fifo_dregs(comedi_device *dev)
{
	comedi_subdevice *s=dev->subdevices+0;
	sampl_t *data;
	int i,n;

	/*
	   Too bad NI didn't have the foresight to return a
	   "fifo empty" value if we read past the end of the
	   FIFO.  It would have made this procedure twice
	   as fast.

	   We can calculate how many samples to transfer.
	   This would save a lot of time.
	*/

	data=((void *)s->cur_trig.data)+s->buf_int_ptr;
	while(1){
		n=(s->cur_trig.data_len-s->buf_int_ptr)/sizeof(sampl_t);
		for(i=0;i<n;i++){
			if(ni_readw(AI_Status_1)&AI_FIFO_Empty_St)
				return;
			*data=ni_readw(ADC_FIFO_Data_Register);
			data++;
			s->buf_int_ptr+=sizeof(sampl_t);
			s->buf_int_count+=sizeof(sampl_t);
		}
		s->buf_int_ptr=0;
		data=s->cur_trig.data;
		comedi_eobuf(dev,s);
	}
}

/*
   used for both cancel ioctl and board initialization

   this is pretty harsh for a cancel, but it works...
 */
static int ni_ai_reset(comedi_device *dev,comedi_subdevice *s)
{
	ungrab_rt_irq(dev,RTirqmask_AI);

	win_out(0x0000,Interrupt_A_Enable_Register);

	win_out(AI_Reset,Joint_Reset_Register);

	win_out(1,ADC_FIFO_Clear);

	/* ai configuration */

	win_out(AI_Configuration_Start,Joint_Reset_Register);

	win_out(0x0000,AI_Command_1_Register); /* reset pulses */
	win_out(0x000d,AI_Mode_1_Register);
	win_out(0x0000,AI_Mode_2_Register);
#if 0
	win_out((1<<6)|0x0000,AI_Mode_3_Register); /* generate FIFO interrupts on half full */
#else
	win_out((0<<6)|0x0000,AI_Mode_3_Register); /* generate FIFO interrupts on non-empty */
#endif
	win_out(0xa4a0,AI_Personal_Register); /* ? */
	win_out(0x032e,AI_Output_Control_Register);
	win_out(0x0060,AI_Trigger_Select_Register); /* trigger source */

	/* this should be done in _ai_modeX() */
	win_out(0x29e0,AI_START_STOP_Select_Register);

	/* the following registers should not be changed:
		Clock_and_FOUT_Register
		AI_Mode_1_Register
		AI_Mode_3_Register
		AI_Personal_Register
		AI_Output_Control_Register
		AI_Trigger_Select_Register
	*/

	win_out(0x3f80,Interrupt_A_Ack_Register); /* clear interrupts */

	win_out(AI_Configuration_End,Joint_Reset_Register);

	return 0;
}

static void ni_load_channelgain_list(comedi_device *dev,unsigned int n_chan,unsigned int *list);


/*
	Mode 0 is immediate
*/
static int ni_ai_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	int i;
	int chan;
	int wsave;
	
	wsave=win_save();

	win_out(1,ADC_FIFO_Clear);

	/* interrupt on errors */
	win_out(0x0020,Interrupt_A_Enable_Register) ;
	
	for(chan=0;chan<it->n_chan;chan++){
		ni_load_channelgain_list(dev,1,it->chanlist+chan);
#if 0
	/* needs start configuration */
	win_out(AI_START_Edge|AI_START_Sync|
		AI_STOP_Select(19)|AI_STOP_Sync,
		AI_START_STOP_Select_Register);
#endif


		win_out(1,AI_Command_1_Register);
	
		/* I don't know how long it takes to access the bus,
		   so shorter loops might cause timeouts */
#define NI_TIMEOUT 1000
		for(i=0;i<NI_TIMEOUT;i++){
			if(!(ni_readw(AI_Status_1)&AI_FIFO_Empty_St)){
				it->data[chan]=ni_readw(ADC_FIFO_Data_Register);
				if(boardtype.adbits==12){
					if(CR_RANGE(it->chanlist[chan])<8)
						it->data[chan]^=0x800;
					it->data[chan]&=0xfff;
				}else{
					if(CR_RANGE(it->chanlist[chan])<8)
						it->data[chan]^=0x8000;
					it->data[chan]&=0xffff;
				}
				break;
			}
			/*udelay(25);*/
		}
		if(i==NI_TIMEOUT)goto timeout;
	}
	win_restore(wsave);
	return chan;

timeout:
	rt_printk("ni_E: timeout 2\n");
	win_restore(wsave);
	return -ETIME;
}

static int ni_ai_mode2(comedi_device *dev,comedi_subdevice *s, comedi_trig *it);
/*
	mode 1 is timed, one channel
*/
static int ni_ai_mode1(comedi_device *dev,comedi_subdevice *s, comedi_trig *it)
{
	it->trigvar1=10;

	return ni_ai_mode2(dev,s,it);

#if 0
	/* use FIFO interrupts to clean FIFO */
	devpriv->lastchan=it->chanlist[0];
	
	win_out(1,ADC_FIFO_Clear);
	win_out(1,Configuration_Memory_Clear);

/*
	mode=getparamval(dev->chinfo[it->chanlist[0]].paramlist,COMEDI_AREF);
*/
	mode=0;

	chan=ni_modebits1[mode]|(it->chanlist[0]&ni_modebits2[mode]);

	ni_writew(chan,Configuration_Memory_High);
	ni_writew(0x8000|devpriv->aip[it->chanlist[0]],
		Configuration_Memory_Low);
	
	/* start configuration */
	win_out(AI_Configuration_Start,Joint_Reset_Register);

	/* stage number of samples */
	win_out((it->n-1)>>16,AI_SC_Load_A_Registers);
	win_out((it->n-1)&0xffff,AI_SC_Load_A_Registers+1);
	
	/* load SC */
	win_out(0x20,AI_Command_1_Register);
	
	/*
	    AI_SI_Special_Trigger_Delay=0
	    AI_Pre_Trigger=0
	  AI_START_STOP_Select_Register:
	    AI_START_Polarity=0 (?)	rising edge
	    AI_START_Edge=1		edge triggered
	    AI_START_Sync=1 (?)		
	    AI_START_Select=0		SI_TC
	    AI_STOP_Polarity=0		rising edge
	    AI_STOP_Edge=0		level
	    AI_STOP_Sync=1		
	    AI_STOP_Select=19		external pin (configuration mem)
	*/
	win_out(AI_START_Edge|AI_START_Sync|
		AI_STOP_Select(19)|AI_STOP_Sync,
		AI_START_STOP_Select_Register);

	win_out((it->trigvar>>16),AI_SI_Load_A_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_A_Registers+1);
	/* AI_SI_Initial_Load_Source=A */
	win_out(0,AI_Mode_2_Register);
	/* load SI */
	win_out(0x200,AI_Command_1_Register);

	/* stage freq. counter into SI B */
	win_out((it->trigvar>>16),AI_SI_Load_B_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_B_Registers+1);

	win_out(1,AI_SI2_Load_A_Register); /* 0,0 does not work. */
	win_out(1,AI_SI2_Load_B_Register);

	/* AI_SI2_Reload_Mode = alternate */
	/* AI_SI2_Initial_Load_Source = A */
	win_out(0x0100,AI_Mode_2_Register);

	/* AI_SI2_Load */
	win_out(0x0800,AI_Command_1_Register);

	/* AI_SI_Initial_Load_Source=0
	   AI_SI_Reload_Mode(0)
	   AI_SI2_Reload_Mode = alternate, AI_SI2_Initial_Load_Source = B */
	win_out(0x0300,AI_Mode_2_Register);

	/* end configuration */
	win_out(AI_Configuration_End,Joint_Reset_Register);

	if(dev->irq){
		/* interrupt on FIFO, errors, SC_TC */
		win_out(0x00a1,Interrupt_A_Enable_Register) ;
	}else{
		/* interrupt on nothing */
		win_out(0x0000,Interrupt_A_Enable_Register) ;
	}
	
	/* AI_SI2_Arm, AI_SI_Arm, AI_DIV_Arm, AI_SC_Arm */
	win_out(0x1540,AI_Command_1_Register);

	/* AI_START1_Pulse */
	win_out(AI_START1_Pulse,AI_Command_2_Register);

	/*dev->busy[0]=it->job;*/
	if(!dev->irq){
		start_polling(dev);
	}
	return 0;
#endif
}

static void ni_load_channelgain_list(comedi_device *dev,unsigned int n_chan,unsigned int *list)
{
	unsigned int chan,range,aref;
	unsigned int i;
	unsigned int hi,lo;

	win_out(1,Configuration_Memory_Clear);

	for(i=0;i<n_chan;i++){
		chan=CR_CHAN(list[i]);
		range=CR_RANGE(list[i]);
		aref=CR_AREF(list[i]);
		
		/* fix the external/internal range differences */
		/* XXX not necessary if sample fixup not done in interrupt */
		range=ni_gainlkup[boardtype.gainlkup][range];
		list[i]=CR_PACK(chan,range,aref);

		hi=ni_modebits1[aref]|(chan&ni_modebits2[aref]);
		ni_writew(hi,Configuration_Memory_High);

		lo=((i==n_chan-1)?0x8000:0) | ((range&0x8)<<5) | (range&0x7);
		ni_writew(lo,Configuration_Memory_Low);
	}

	/* prime the channel/gain list */

	win_out(1,AI_Command_1_Register);
	for(i=0;i<40;i++){
		if(!(ni_readw(AI_Status_1)&AI_FIFO_Empty_St)){
			win_out(1,ADC_FIFO_Clear);
			return;
		}
		udelay(25);
	}
	rt_printk("ni_E: timeout 1\n");
}


/*
	mode 2 is timed, multi-channel
*/
static int ni_ai_mode2(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	int wsave;

	if(it->flags&TRIG_RT)
		grab_rt_irq(dev,RTirqmask_AI);

	wsave=win_save();

	win_out(1,ADC_FIFO_Clear);

	ni_load_channelgain_list(dev,it->n_chan,it->chanlist);
	
	/* start configuration */
	win_out(AI_Configuration_Start,Joint_Reset_Register);

	/* stage number of scans */
	win_out((it->n-1)>>16,AI_SC_Load_A_Registers);
	win_out((it->n-1)&0xffff,AI_SC_Load_A_Registers+1);
	
	/* load SC (Scan Count) */
	win_out(0x20,AI_Command_1_Register);
	
	/*
	    AI_SI_Special_Trigger_Delay=0
	    AI_Pre_Trigger=0
	  AI_START_STOP_Select_Register:
	    AI_START_Polarity=0 (?)	rising edge
	    AI_START_Edge=1		edge triggered
	    AI_START_Sync=1 (?)		
	    AI_START_Select=0		SI_TC
	    AI_STOP_Polarity=0		rising edge
	    AI_STOP_Edge=0		level
	    AI_STOP_Sync=1		
	    AI_STOP_Select=19		external pin (configuration mem)
	 */
	win_out(AI_START_Edge|AI_START_Sync|
		AI_STOP_Select(19)|AI_STOP_Sync,
		AI_START_STOP_Select_Register);

	win_out((it->trigvar>>16),AI_SI_Load_A_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_A_Registers+1);
	/* AI_SI_Initial_Load_Source=A */
	win_out(0,AI_Mode_2_Register);
	/* load SI */
	win_out(0x200,AI_Command_1_Register);

	/* stage freq. counter into SI B */
	win_out((it->trigvar>>16),AI_SI_Load_B_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_B_Registers+1);

	win_out(it->trigvar1,AI_SI2_Load_A_Register); /* 0,0 does not work. */
	win_out(it->trigvar1,AI_SI2_Load_B_Register);

	/* AI_SI2_Reload_Mode = alternate */
	/* AI_SI2_Initial_Load_Source = A */
	win_out(0x0100,AI_Mode_2_Register);

	/* AI_SI2_Load */
	win_out(0x0800,AI_Command_1_Register);

	/* AI_SI_Initial_Load_Source=0
	   AI_SI_Reload_Mode(0)
	   AI_SI2_Reload_Mode = alternate, AI_SI2_Initial_Load_Source = B */
	win_out(0x0300,AI_Mode_2_Register);

	if(dev->irq){
		int bits;

		/* interrupt on FIFO, errors, SC_TC */
		bits=0x00a1;

		if(it->flags&TRIG_WAKE_EOS || s->cb_mask&COMEDI_CB_EOS){
			/*generate FIFO interrupts on non-empty */
			win_out(AI_FIFO_Mode_NE|0x0000,AI_Mode_3_Register);
#if 0
			bits|=AI_START_Interrupt_Enable|AI_STOP_Interrupt_Enable;
#endif
		}else{
			/*generate FIFO interrupts on non-empty */
			win_out(AI_FIFO_Mode_HF|0x0000,AI_Mode_3_Register);
		}
		win_out(0x3f80,Interrupt_A_Ack_Register); /* clear interrupts */

		win_out(bits,Interrupt_A_Enable_Register) ;
	}else{
		/* interrupt on nothing */
		win_out(0x0000,Interrupt_A_Enable_Register) ;

		/* XXX start polling if necessary */
	}

#ifdef DEBUG
rt_printk("end config\n");
#endif
	/* end configuration */
	win_out(AI_Configuration_End,Joint_Reset_Register);
	
	/* AI_SI2_Arm, AI_SI_Arm, AI_DIV_Arm, AI_SC_Arm */
	win_out(0x1540,AI_Command_1_Register);

	/* AI_START1_Pulse */
	win_out(AI_START1_Pulse,AI_Command_2_Register);
#ifdef DEBUG
rt_printk("START1 pulse\n");
#endif

	/* XXX hack alert */
	s->cur_chan=s->cur_trig.n_chan*sizeof(sampl_t);

	win_restore(wsave);
	return 0;
}

/*
	mode 4 is external trigger for scans, timer for samples
	in a scan
*/
static int ni_ai_mode4(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	int wsave;

	if(it->flags&TRIG_RT)
		grab_rt_irq(dev,RTirqmask_AI);

	wsave=win_save();

	win_out(1,ADC_FIFO_Clear);

	ni_load_channelgain_list(dev,it->n_chan,it->chanlist);
	
	/* start configuration */
	win_out(AI_Configuration_Start,Joint_Reset_Register);

	/* stage number of scans */
	win_out((it->n-1)>>16,AI_SC_Load_A_Registers);
	win_out((it->n-1)&0xffff,AI_SC_Load_A_Registers+1);
	
	/* load SC (Scan Count) */
	win_out(0x20,AI_Command_1_Register);
	
	/*
	    AI_SI_Special_Trigger_Delay=0
	    AI_Pre_Trigger=0
	  AI_START_STOP_Select_Register:
	    AI_START_Polarity=0 (?)	rising edge
	    AI_START_Edge=1		edge triggered
	    AI_START_Sync=1 (?)		
	    AI_START_Select=1		PFI0
	    AI_STOP_Polarity=0		rising edge
	    AI_STOP_Edge=0		level
	    AI_STOP_Sync=1		
	    AI_STOP_Select=19		external pin (configuration mem)
	*/
	win_out(AI_START_Edge|AI_START_Sync|AI_START_Select(1)|
		AI_STOP_Select(19)|AI_STOP_Sync,
		AI_START_STOP_Select_Register);
	
#if 0
	win_out((it->trigvar>>16),AI_SI_Load_A_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_A_Registers+1);
	/* AI_SI_Initial_Load_Source=A */
	win_out(0,AI_Mode_2_Register);
	/* load SI */
	win_out(0x200,AI_Command_1_Register);

	/* stage freq. counter into SI B */
	win_out((it->trigvar>>16),AI_SI_Load_B_Registers);
	win_out((it->trigvar&0xffff),AI_SI_Load_B_Registers+1);
#endif

	win_out(it->trigvar1,AI_SI2_Load_A_Register); /* 0,0 does not work. */
	win_out(it->trigvar1,AI_SI2_Load_B_Register);

	/* AI_SI2_Reload_Mode = alternate */
	/* AI_SI2_Initial_Load_Source = A */
	win_out(0x0100,AI_Mode_2_Register);

	/* AI_SI2_Load */
	win_out(0x0800,AI_Command_1_Register);

	/* AI_SI_Initial_Load_Source=0
	   AI_SI_Reload_Mode(0)
	   AI_SI2_Reload_Mode = alternate, AI_SI2_Initial_Load_Source = B */
	win_out(0x0300,AI_Mode_2_Register);

	if(dev->irq){
		int bits;

		/* interrupt on FIFO, errors, SC_TC */
		bits=0x00a1;

		if(it->flags&TRIG_WAKE_EOS || s->cb_mask&COMEDI_CB_EOS){
			/* generate FIFO interrupts on non-empty */
			win_out(AI_SI2_Source_Select|AI_FIFO_Mode_NE|0x0000,AI_Mode_3_Register);
		}else{
			/* generate FIFO interrupts on half-full */
			win_out(AI_SI2_Source_Select|AI_FIFO_Mode_HF|0x0000,AI_Mode_3_Register);
		}

		/* clear interrupts */
		win_out(0x3f80,Interrupt_A_Ack_Register);

		win_out(bits,Interrupt_A_Enable_Register) ;
	}else{
		/* interrupt on nothing */
		win_out(0x0000,Interrupt_A_Enable_Register) ;

		/* XXX start polling if necessary */
	}
	
	/* end configuration */
	win_out(AI_Configuration_End,Joint_Reset_Register);

	/* AI_SI2_Arm, AI_DIV_Arm, AI_SC_Arm */
	win_out(0x1500,AI_Command_1_Register);

	/* AI_START1_Pulse */
	win_out(AI_START1_Pulse,AI_Command_2_Register);

	/* XXX hack alert */
	s->cur_chan=s->cur_trig.n_chan*sizeof(sampl_t);

	win_restore(wsave);
	return 0;
}


static void ni_ao_fifo_load(comedi_device *dev,comedi_subdevice *s,
		sampl_t *data,int n)
{
	int i;

	for(i=0;i<n;i++){
		ni_writew(data[i],DAC_FIFO_Data);
	}
}


/*
 *  There's a small problem if the FIFO gets really low and we
 *  don't have the data to fill it.  Basically, if after we fill
 *  the FIFO with all the data available, the FIFO is _still_
 *  less than half full, we never clear the interrupt.  If the
 *  IRQ is in edge mode, we never get another interrupt, because
 *  this one wasn't cleared.  If in level mode, we get flooded
 *  with interrupts that we can't fulfill, because nothing ever
 *  gets put into the buffer.
 *
 *  This kind of situation is recoverable, but it is easier to
 *  just pretend we had a FIFO underrun, since there is a good
 *  chance it will happen anyway.  This is _not_ the case for
 *  RT code, as RT code might purposely be running close to the
 *  metal.  Needs to be fixed eventually.
 */
static int ni_ao_fifo_half_empty(comedi_device *dev,comedi_subdevice *s)
{
	int n,m;

	n=(s->buf_int_count-s->buf_user_count)/sizeof(sampl_t);
	if(n==0)return 0;
	if(n>boardtype.ao_fifo_depth/2)
		n=boardtype.ao_fifo_depth/2;

	if(s->buf_int_ptr+n*sizeof(sampl_t)>s->cur_trig.data_len){
		m=(s->cur_trig.data_len-s->buf_int_ptr)/sizeof(sampl_t);
		ni_ao_fifo_load(dev,s,((void *)(s->cur_trig.data))+s->buf_int_ptr,m);
		s->buf_int_count+=m*sizeof(sampl_t);
		s->buf_int_ptr=0;
		n-=m;
	}
	ni_ao_fifo_load(dev,s,((void *)(s->cur_trig.data))+s->buf_int_ptr,n);
	s->buf_int_count+=n*sizeof(sampl_t);
	s->buf_int_ptr+=n*sizeof(sampl_t);

	comedi_bufcheck(dev,s);

	return 1;
}

static int ni_ao_prep_fifo(comedi_device *dev,comedi_subdevice *s)
{
	int n;

	/* reset fifo */
	win_out(0,DAC_FIFO_Clear);

	/* load some data */
	n=(s->buf_int_count-s->buf_user_count)/sizeof(sampl_t);
	if(n==0)return 0;
	if(n>boardtype.ao_fifo_depth)
		n=boardtype.ao_fifo_depth;

	ni_ao_fifo_load(dev,s,((void *)(s->cur_trig.data))+s->buf_int_ptr,n);
	s->buf_int_count+=n*sizeof(sampl_t);
	s->buf_int_ptr+=n*sizeof(sampl_t);

	return n;
}


#if 0
static void AO_Reset_All(comedi_device *dev)
{
	win_out(AO_Configuration_Start,Joint_Reset_Register);

	win_out(AO_Disarm,AO_Command_1_Register);
	devpriv->cmd1=0;

	win_out(0,Interrupt_B_Enable_Register);

	win_out(AO_BC_Source_Select(1),AO_Personal_Register);

	win_out(0x3f98,Interrupt_B_Ack_Register);

	win_out(AO_Configuration_End,Joint_Reset_Register);
}

static void AO_Board_Personalize(comedi_device *dev)
{
	win_out(AO_Configuration_Start,Joint_Reset_Register);

	win_out(0x1430,AO_Personal_Register);

	win_out(0x1b20,Clock_and_FOUT_Register);

	win_out(0,AO_Output_Control_Register);

	win_out(0,AO_START_Select_Register);

	win_out(AO_Configuration_End,Joint_Reset_Register);
}

static void AO_LDAC_Source_And_Update_Mode(comedi_device *dev)
{
	win_out(AO_Configuration_Start,Joint_Reset_Register);

	devpriv->ao_cmd1&=~(AO_LDAC0_Source_Select|AO_DAC0_Update_Mode|AO_LDAC1_Source_Select|AO_DAC1_Update_Mode);
	win_out(devpriv->ao_cmd1,AO_Command_1_Register);

	win_out(AO_Configuration_End,Joint_Reset_Register);
}

static void AO_Triggering(comedi_device *dev)
{
	win_out(AO_Configuration_Start,Joint_Reset_Register);

	devpriv->ao_mode1|=AO_Trigger_Once;
	win_out(devpriv->ao_mode1,AO_Mode_1_Register);

	devpriv->ao_trigger_select&=~(AO_START1_Select(-1)|AO_START1_Polarity);
	devpriv->ao_trigger_select|=AO_START1_Edge|AO_START1_Sync;
	win_out(devpriv->ao_trigger_select,AO_Trigger_Select_Register);

	devpriv->ao_mode3&=~AO_Trigger_Length;
	win_out(devpriv->ao_mode3,AO_Mode_3_Register);
	
	win_out(AO_Configuration_End,Joint_Reset_Register);
}

static void AO_Counting(comedi_device *dev)
{
	win_out(AO_Configuration_Start,Joint_Reset_Register);

	devpriv->ao_mode1&=~AO_Continuous;
	win_out(devpriv->ao_mode1,AO_Mode_1_Register);

	devpriv->ao_mode2&=~AO_BC_Initial_Load_Source;
	win_out(devpriv->ao_mode2,AO_Mode_2_Register);

	win_out(0,AO_BC_Load_A_Register_High);
	win_out(4,AO_BC_Load_A_Register_Low);
	win_out(devpriv->ao_cmd1|AO_BC_Load,AO_Command_1_Register);

	devpriv->ao_mode2&=~AO_UC_Initial_Load_Source;
	win_out(devpriv->ao_mode2,AO_Mode_2_Register);
	
	win_out(0,AO_UC_Load_A_Register_High);
	win_out(3000,AO_UC_Load_A_Register_Low);
	win_out(devpriv->ao_cmd1|AO_UC_Load,AO_Command_1_Register);

	win_out(0,AO_UC_Load_A_Register_High);
	win_out(2999,AO_UC_Load_A_Register_Low);

	win_out(AO_Configuration_End,Joint_Reset_Register);
}

static void AO_Updating(comedi_device *dev)
{
	win_out(AO_Configuration_Start,Joint_Reset_Register);

	devpriv->ao_cmd2&=~AO_BC_Gate_Enable;
	win_out(devpriv->ao_cmd2,AO_Command_2_Register);

	devpriv->ao_mode1&=~(AO_UPDATE_Source_Select(-1)|AO_UPDATE_Source_Polarity);
	devpriv->ao_mode1&=~(AO_UI_Source_Select(-1)|AO_UI_Source_Polarity);
	win_out(devpriv->ao_mode1,AO_Mode_1_Register);

	devpriv->ao_mode2&=~(AO_UI_Initial_Load_Source|AO_UI_Reload_Mode);
	win_out(devpriv->ao_mode2,AO_Mode_2_Register);

	win_out(0,AO_UI_Load_A_Register_High);
	win_out(1,AO_UI_Load_A_Register_Low);
	win_out(devpriv->ao_cmd1|AO_UI_Load,AO_Command_1_Register);

	win_out(0,AO_UI_Load_A_Register_High);
	win_out(0x9c40,AO_UI_Load_A_Register_Low);

	win_out(AO_Configuration_End,Joint_Reset_Register);
}

static void AO_Channels(comedi_device *dev)
{
	win_out(AO_Configuration_Start,Joint_Reset_Register);

	devpriv->ao_mode1&=~AO_Multiple_Channels;
	win_out(devpriv->ao_mode1,AO_Mode_1_Register);

	win_out(AO_Number_Of_Channels(1),AO_Output_Control_Register);

	win_out(AO_Configuration_End,Joint_Reset_Register);
}



#endif




static int ni_ao_mode0(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	unsigned int data;
	unsigned int conf;
	unsigned int chan;
	unsigned int range;
	
	data=it->data[0];
	chan=CR_CHAN(it->chanlist[0]);

	conf=chan<<8;
	
	/* XXX check range with current range in flaglist[chan] */
	/* should update calibration if range changes (ick) */

	range = CR_RANGE(it->chanlist[0]);
	conf |= (range&1);
	conf |= (range&2)<<1;
	
	/* not all boards can deglitch, but this shouldn't hurt */
	if(it->flags & TRIG_DEGLITCH)
		conf |= 2;

	/* analog reference */
	/* AREF_OTHER connects AO ground to AI ground, i think */
	conf |= (CR_AREF(it->chanlist[0])==AREF_OTHER)? 8 : 0;

	ni_writew(conf,AO_Configuration);

	if(range&1)data^=0x800;
	
	ni_writew(data,(chan)? DAC1_Direct_Data : DAC0_Direct_Data);
	
	return 1;
}

static int ni_ao_mode2(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	unsigned int conf;
	unsigned int chan;
	unsigned int range;
	
	chan=CR_CHAN(it->chanlist[0]);

	conf=chan<<8;
	
	/* XXX check range with current range in flaglist[chan] */
	/* should update calibration if range changes (ick) */

	range = CR_RANGE(it->chanlist[0]);
	conf |= (range&1);
	conf |= (range&2)<<1;
	
	/* not all boards can deglitch, but this shouldn't hurt */
	if(it->flags & TRIG_DEGLITCH)
		conf |= 2;

	/* analog reference */
	/* AREF_OTHER connects AO ground to AI ground, i think */
	conf |= (CR_AREF(it->chanlist[0])==AREF_OTHER)? 8 : 0;

	win_out(AO_Disarm,AO_Command_1_Register);

	win_out(conf,AO_Configuration);

	/* user is supposed to write() to buffer before triggering */
	if(ni_ao_prep_fifo(dev,s)==0)
		return -EIO;

	win_out(AO_Configuration_Start,Joint_Reset_Register);

	devpriv->ao_mode1|=AO_Trigger_Once;
	win_out(devpriv->ao_mode1,AO_Mode_1_Register);
	devpriv->ao_trigger_select&=~(AO_START1_Polarity|AO_START1_Select(-1));
	devpriv->ao_trigger_select|=AO_START1_Edge|AO_START1_Sync;
	win_out(devpriv->ao_trigger_select,AO_Trigger_Select_Register);
	devpriv->ao_mode3&=~AO_Trigger_Length;
	win_out(devpriv->ao_mode3,AO_Mode_3_Register);

	if(it->n==0){
		devpriv->ao_mode1|=AO_Continuous;
printk("setting continuous\n");
	}else{
		devpriv->ao_mode1&=~AO_Continuous;
printk("unsetting continuous\n");
	}
	win_out(devpriv->ao_mode1,AO_Mode_1_Register);
	devpriv->ao_mode2&=~AO_BC_Initial_Load_Source;
	win_out(devpriv->ao_mode2,AO_Mode_2_Register);
	if(it->n==0){
		win_out(0xff,AO_BC_Load_A_Register_High);
		win_out(0xffff,AO_BC_Load_A_Register_Low);
	}else{
		win_out(0,AO_BC_Load_A_Register_High);
		win_out(0,AO_BC_Load_A_Register_Low);
	}
	win_out(AO_BC_Load,AO_Command_1_Register);
	devpriv->ao_mode2&=~AO_UC_Initial_Load_Source;
	win_out(devpriv->ao_mode2,AO_Mode_2_Register);
	if(it->n==0){
		win_out(0xff,AO_UC_Load_A_Register_High);
		win_out(0xffff,AO_UC_Load_A_Register_Low);
		win_out(AO_UC_Load,AO_Command_1_Register);
		win_out(0xff,AO_UC_Load_A_Register_High);
		win_out(0xffff,AO_UC_Load_A_Register_Low);
	}else{
		win_out(0,AO_UC_Load_A_Register_High);
		win_out(0,AO_UC_Load_A_Register_Low);
		win_out(AO_UC_Load,AO_Command_1_Register);
		win_out((it->n-1)>>16,AO_UC_Load_A_Register_High);
		win_out((it->n-1)&0xffff,AO_UC_Load_A_Register_Low);
	}

	devpriv->ao_cmd2&=~AO_BC_Gate_Enable;
	ni_writew(devpriv->ao_cmd2,AO_Command_2);
	devpriv->ao_mode1&=~(AO_UI_Source_Select(0x1f)|AO_UI_Source_Polarity);
	win_out(devpriv->ao_mode1,AO_Mode_1_Register);
	devpriv->ao_mode2&=~(AO_UI_Reload_Mode(3)|AO_UI_Initial_Load_Source);
	win_out(devpriv->ao_mode2,AO_Mode_2_Register);
	win_out(0,AO_UI_Load_A_Register_High);
	win_out(1,AO_UI_Load_A_Register_Low);
	win_out(AO_UI_Load,AO_Command_1_Register);
	win_out((it->trigvar>>16),AO_UI_Load_A_Register_High);
	win_out((it->trigvar&0xffff),AO_UI_Load_A_Register_Low);

	devpriv->ao_mode1&=~AO_Multiple_Channels;
	win_out(devpriv->ao_mode1,AO_Mode_1_Register);
	win_out(AO_UPDATE_Output_Select(1),AO_Output_Control_Register);

	win_out(AO_DAC0_Update_Mode|AO_DAC1_Update_Mode,AO_Command_1_Register);

	devpriv->ao_mode3|=AO_Stop_On_Overrun_Error;
	win_out(devpriv->ao_mode3,AO_Mode_3_Register);

devpriv->ao_mode2|=AO_FIFO_Mode(1);
	devpriv->ao_mode2&=~AO_FIFO_Retransmit_Enable;
	win_out(devpriv->ao_mode2,AO_Mode_2_Register);

	win_out(AO_Configuration_End,Joint_Reset_Register);

	win_out(devpriv->ao_mode3|AO_Not_An_UPDATE,AO_Mode_3_Register);
	win_out(devpriv->ao_mode3,AO_Mode_3_Register);

	/* wait for DACs to be loaded */
	udelay(100);

	win_out(devpriv->ao_cmd1|AO_UI_Arm|AO_UC_Arm|AO_BC_Arm|AO_DAC1_Update_Mode|AO_DAC0_Update_Mode,
		AO_Command_1_Register);

	win_out(AO_FIFO_Interrupt_Enable|AO_Error_Interrupt_Enable,Interrupt_B_Enable_Register);


	ni_writew(devpriv->ao_cmd2|AO_START1_Pulse,AO_Command_2);
	
	return 0;
}



/*
	digital i/o

	assume digital output for now
*/
static int ni_dio(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	int data,mask;
	int i;
	int temp;

	/* rt stuff */
	temp=win_save();

	if(it->flags & TRIG_CONFIG){
		data=s->io_bits;
		for(i=0;i<it->n_chan;i++){
			mask=1<<CR_CHAN(it->chanlist[i]);
			data&= ~mask;
			if(it->data[i])
				data|=mask;
		}
		s->io_bits=data;
		win_out(s->io_bits,DIO_Control_Register);
	}else{
		if(it->flags & TRIG_WRITE){
			do_pack(&s->state,it);
			win_out(s->state,DIO_Output_Register);
		}else{
			data=win_in(DIO_Input_Register);
			di_unpack(data,it);
		}
	}

	win_restore(temp);

	return it->n_chan;
}

#if 0
static int ni_dsp(comedi_device *dev,comedi_param *it)
{
	switch(it->pnum){
	case COMEDI_IOBITS:
		if(it->pval&~0xff)return -EINVAL;
		devpriv->dio=it->pval;
		win_out(devpriv->dio,DIO_Control_Register);
		changeparam(&dev->chinfo[it->chan].paramlist,COMEDI_IOBITS,0,
			it->pval);
		return 0;
	default:
		return -EINVAL;
	}
}
#endif

/*
   XXX this assumes that if there isn't anything in the fifo,
   then we can turn off polling.  this is *not* a good idea,
   but it will do for now.
   Note: there is always the possibility that the poll
   routine will steal a sample from a soft triggered AI.
*/

#if 0
static void ni_E_poll(comedi_device *dev)
{
	sampl_t data;
	int wake=0;
	int i;
	int status;
#ifdef DEBUG
	int n=0;
#endif
	
	status=ni_readw(AI_Status_1);
#ifdef DEBUG
	printk("ni_E: polling... jiffies=%d status=0x%04x ",jiffies,status);
#endif

	if(status&AI_FIFO_Half_Full_St){
#ifdef DEBUG
		printk("fifo half full ");
		n+=boardtype.ai_fifo_depth/2;
#endif
		for(i=0;i<boardtype.ai_fifo_depth/2;i++){
			data=ni_readw(ADC_FIFO_Data_Register);
			if(!(devpriv->aip[devpriv->lastchan]&0x100))
				data^=0x800;
			data&=0xfff; /* incompatible with 16 bit cards */
			report_sample(dev,data);
		}
		wake=1;
	}
	while(!(ni_readw(AI_Status_1)&AI_FIFO_Empty_St)){
		data=ni_readw(ADC_FIFO_Data_Register);
		if(!(devpriv->aip[devpriv->lastchan]&0x100))
			data^=0x800;
		data&=0xfff; /* incompatible with 16 bit cards */
		report_sample(dev,data);
		wake=1;
#ifdef DEBUG
		n++;
#endif
	}
#ifdef DEBUG
	printk("samples=%d\n",n);
#endif

	if(wake)comedi_done(dev,dev->subdevices+0);
	else{
		stop_polling(dev);
		/*dev->busy[0]=0;*/
	}
}
#endif


static int ni_E_init(comedi_device *dev,comedi_devconfig *it)
{
	comedi_subdevice *s;
	
	dev->n_subdevices=7;
	
	if(alloc_subdevices(dev)<0)
		return -ENOMEM;
	
	/* analog input subdevice */

	s=dev->subdevices+0;
	s->type=COMEDI_SUBD_AI;
	s->subdev_flags=SDF_READABLE|SDF_RT|SDF_GROUND|SDF_COMMON|SDF_DIFF|SDF_OTHER;
#if 0
	s->subdev_flags|=SDF_DITHER;
#endif
	s->n_chan=boardtype.n_adchan;
	s->len_chanlist=512;	/* XXX is this the same for PCI-MIO ? */
	s->maxdata=(1<<boardtype.adbits)-1;
	s->timer_type=TIMER_atmio;
	s->range_type=ni_range_lkup[boardtype.gainlkup];
	s->trig[0]=ni_ai_mode0;
	s->trig[1]=ni_ai_mode1;		/* not true mode 1 */
	s->trig[2]=ni_ai_mode2;
	s->trig[4]=ni_ai_mode4;
	s->cancel=ni_ai_reset;
	
	/* analog output subdevice */
	/* XXX what about boards without ao? */

	s=dev->subdevices+1;
	if(boardtype.n_aochan){
		s->type=COMEDI_SUBD_AO;
		s->subdev_flags=SDF_WRITEABLE|SDF_RT|SDF_DEGLITCH|SDF_GROUND|SDF_OTHER;
		s->n_chan=boardtype.n_aochan;
		s->maxdata=(1<<boardtype.aobits)-1;
		s->timer_type=TIMER_atmio;
		s->range_type=RANGE_ni_E_ao_ext;	/* XXX wrong for some boards */
		s->trig[0]=ni_ao_mode0;
		s->trig[2]=ni_ao_mode2;
	}else{
		s->type=COMEDI_SUBD_UNUSED;
	}
	
	/* digital i/o subdevice */
	
	s=dev->subdevices+2;
	s->type=COMEDI_SUBD_DIO;
	s->subdev_flags=SDF_WRITEABLE|SDF_READABLE|SDF_RT;
	s->n_chan=8;
	s->maxdata=1;
	s->range_type=RANGE_digital;
	s->io_bits=0;		/* all bits input */
	s->trig[0]=ni_dio;

	/* dio setup */
	win_out(s->io_bits,DIO_Control_Register);
	
	/* 8255 device */
	s=dev->subdevices+3;
	if(boardtype.has_8255){
		subdev_8255_init(dev,s,ni_8255_callback,dev);
	}else{
		s->type=COMEDI_SUBD_UNUSED;
		s->trig[0]=NULL;
	}
	/* XXX */
	
	/* timer/counter device */
	s=dev->subdevices+4;
	s->type=COMEDI_SUBD_UNUSED;
	s->trig[0]=NULL;
	/* XXX */
	
	/* calibration subdevice -- ai and ao */
	s=dev->subdevices+5;
	s->type=COMEDI_SUBD_CALIB;
	s->subdev_flags=SDF_WRITEABLE|SDF_INTERNAL;
	s->n_chan=(boardtype.caldactype==caldac_mb88341)?12:9;
	s->maxdata=0xfff;
	s->trig[0]=ni_calib;
	
	/* EEPROM */
	s=dev->subdevices+6;
	s->type=COMEDI_SUBD_MEMORY;
	s->subdev_flags=SDF_READABLE|SDF_INTERNAL;
	s->n_chan=512;
	s->maxdata=0xff;
	s->trig[0]=ni_eeprom;
	
	/* ai configuration */
	ni_ai_reset(dev,dev->subdevices+0);
	win_out(0x1ba0,Clock_and_FOUT_Register);


	/* analog output configuration */
	
	devpriv->ao0p=0x0000;
	ni_writew(devpriv->ao0p,AO_Configuration);
	devpriv->ao1p=0x0100;
	ni_writew(devpriv->ao1p,AO_Configuration);
	win_out(AO_Configuration_Start,Joint_Reset_Register);
	win_out(AO_Disarm,AO_Command_1_Register);
	win_out(0,Interrupt_B_Enable_Register);
	win_out(0x0010,AO_Personal_Register);
	ni_writew(0x3f98,Interrupt_B_Ack);
	win_out(0x1430,AO_Personal_Register);
	win_out(0,AO_Output_Control_Register);
	win_out(0,AO_Start_Select_Register);
	devpriv->ao_cmd1=0;
	win_out(devpriv->ao_cmd1,AO_Command_1_Register);
	devpriv->ao_cmd2=0;
	devpriv->ao_mode1=0;
	devpriv->ao_mode2=0;
	devpriv->ao_mode3=0;
	devpriv->ao_trigger_select=0;

        if(dev->irq){
                win_out((IRQ_POLARITY<<0) |  /* polarity : active high */
                        (0<<1) |  /* no interrupt on 3 pins */
                        (1<<11) |  /* enable int A */
                        (1<<15) |  /* enable int B */
                        (interrupt_pin(dev->irq)<<8) |  /* interrupt output pin A */
                        (interrupt_pin(dev->irq)<<12) ,  /* interrupt output pin B */
                        Interrupt_Control_Register
                );
        }

	
	dev->rem=ni_rem;

	printk("\n");

	return 1;
}



/* called when driver is removed */
static int ni_rem(comedi_device *dev)
{
	printk("comedi%d: ni_E: remove\n",dev->minor);
	
	/* board-specific cleanup */
	ni_E_rem_stage2(dev);
	
	return 0;
}


/*
	presents the EEPROM as a subdevice
*/

static int ni_eeprom(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	int i;
	
	for(i=0;i<it->n_chan;i++)
		it->data[i]=ni_read_eeprom(dev,CR_CHAN(it->chanlist[i]));

	return i;
}

/*
	reads bytes out of eeprom
*/

static int ni_read_eeprom(comedi_device *dev,int addr)
{
	int bit;
	int bitstring;

	bitstring=0x0300|((addr&0x100)<<3)|(addr&0xff);
	ni_writeb_p(0x04,Serial_Command);
	for(bit=0x8000;bit;bit>>=1){
		ni_writeb_p(0x04|((bit&bitstring)?0x02:0),Serial_Command);
		ni_writeb_p(0x05|((bit&bitstring)?0x02:0),Serial_Command);
	}
	bitstring=0;
	for(bit=0x80;bit;bit>>=1){
		ni_writeb_p(0x04,Serial_Command);
		ni_writeb_p(0x05,Serial_Command);
		bitstring|=((ni_readb_p(Status)&0x01)?bit:0);
	}
	ni_writeb_p(0x00,Serial_Command);
	
	return bitstring;
}

static void ni_write_caldac(comedi_device *dev,int addr,int val);
/*
	calibration subdevice
*/
static int ni_calib(comedi_device *dev,comedi_subdevice *s,comedi_trig *it)
{
	ni_write_caldac(dev,CR_CHAN(it->chanlist[0]),it->data[0]);

	return 1;
}

static void ni_write_caldac(comedi_device *dev,int addr,int val)
{
	int loadbit,bits,bit,bitstring;
	
	switch(boardtype.caldactype){
	case caldac_mb88341:
		/*
		   MB 88341
		   Note that address bits are reversed.  Thanks to
		   Ingo Keen for noticing this.

		   Note also that the 88341 expects address values from
		   1-12, whereas we use channel numbers 0-11.  The NI
		   docs use 1-12, also, so be careful here.
		*/
		addr++;
		bits=12;
		bitstring=((addr&0x1)<<11) |
			  ((addr&0x2)<<9)  |
			  ((addr&0x4)<<7)  |
			  ((addr&0x8)<<5)  |
			  (val&0xff);
		loadbit=0x08;
		break;
	case caldac_dac8800:
		if(addr==8){
			/*
			   DAC 8043
			*/
			bits=12;
			bitstring=val&0xfff;
			loadbit=0x10;
		}else{
			/*
			   DAC 8800
			*/
			bits=11;
			bitstring=((addr&0x7)<<8)|(val&0xff);
			loadbit=0x08;
		}
		break;
	default:
		return;
	}

	for(bit=1<<(bits-1);bit;bit>>=1){
		ni_writeb(((bit&bitstring)?0x02:0),Serial_Command);
		ni_writeb(1|((bit&bitstring)?0x02:0),Serial_Command);
	}
	ni_writeb(loadbit,Serial_Command);
	ni_writeb(0,Serial_Command);
}



/*
 *
 *  Programmable Function Inputs
 *
 */

#if 0

static void pfi_setup(comedi_device *dev)
{
	/* currently, we don't output any signals, thus, all
	   the PFI's are input */
	
	win_out(IO_Bidirection_Pin_Register,0);
}



/*
 *
 *  General Purpose Counter/Timer section
 *
 */


/* event counting */

int gpct_start(comedi_device *dev,int chan)
{
	/* select load source */
	Gi_Load_Source_Select = 0;
	
	/* load initial counter value */
	Gi_Load_A = 0;
	
	/* strobe */
	Gi_Load=1;

/* command reg */
	Gi_Up_Down = 1; /* up counting */
	Gi_Bank_Switch_Enable = 0;
	Gi_Bank_Switch_Mode = 0;

/* input select reg */
	Gi_Source_Select = PFIn;
	Gi_Source_Polarity = 0; /* rising edges */
	Gi_Gate_Select = 31; /* logic low */
	Gi_OR_Gate = 0;
	Gi_Output_Polarity = 1; /* active low */
	Gi_Gate_Select_Load_Source = 0;

/* mode reg */
	Gi_Gate_Polarity = 0; /* disable inversion */
	Gi_Loading_On_Gate = 0;
	Gi_Output_Mode = 1; /* one clock cycle output */
	Gi_Loading_On_TC = 0;
	Gi_Gating_Mode = 1;
	Gi_Gate_On_Both_Edges = 0;
	Gi_Trigger_Mode_For_Edge_Gate = 2;
	Gi_Stop_Mode = 0;
	Gi_Counting_Once = 0;
	
	
/* int enable reg -- ignore */
	
/*
	Gi_TC_Interrupt_Enable = 0;
	Gi_Gate_Interrupt_Enable = 0;
*/
}

static int gpct_sp(comedi_device *dev,comedi_param *it)
{
	switch(it->pnum){
	case COMEDI_CTR_INPUT:
	{
		/* which input source? */
		int pval=it->pval;
		
		if(pval<0 || pval>=10)return -EINVAL;
		changeparam(&dev->chinfo[it->chan].paramlist,COMEDI_CTR_INPUT,
			0,pval);
		/* XXX set it */
		return 0;
	}
	case COMEDI_CTR_POLARITY:
		/* count rising or falling edges? */
		if(it->pval<0 || it->pval>=4)return -EINVAL;
		changeparam(&dev->chinfo[it->chan].paramlist,COMEDI_AREF,
			0,it->pval);
		/* XXX set it */
		return 0;
	case COMEDI_COUNT:
		changeparam(&dev->chinfo[it->chan].paramlist,COMEDI_COUNT,
			0,it->pval);  /* irrelevant... */
		devpriv->gpct[it->chan-boardtype->counteroffset]=it->pval;
		return 0;
	}
	return -EINVAL;
}

#endif

static int ni_8255_callback(int dir,int port,int data,void *arg)
{
	comedi_device *dev=arg;

	if(dir){
		ni_writeb(25+2*port,data);
		return 0;
	}else{
		return ni_readb(25+2*port);
	}
}

