/*
    module/ni-dio.c
    driver for National Instruments PCI-DIO-96

    COMEDI - Linux Control and Measurement Device Interface
    Copyright (C) 1999 David A. Schleef <ds@stm.lbl.gov>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include <linux/kernel.h>
#include <comedi_module.h>
#include <linux/errno.h>
#include <linux/ioport.h>
#include <asm/io.h>
#include <linux/malloc.h>
#include <linux/delay.h>
#include <linux/pci.h>
#include <8255.h>

/* general debugging messages */
#define DEBUG


#define PCI_VENDOR_ID_NATINST	0x1093
#define PCI_DEVICE_ID_DIO96	0x0160

#define PCI_DIO_SIZE 4096
#define PCI_MITE_SIZE 4096

#define NIDIO_A 0
#define NIDIO_B 4
#define NIDIO_C 8
#define NIDIO_D 12

static int nidio96_rem(comedi_device *dev);

typedef struct{
	struct pci_dev *pci_device;
	unsigned long daq_phys_addr;
	unsigned long mite_phys_addr;
	void *daq_io_addr;
	void *mite_io_addr;
	int boardtype;
	int dio;
	int aip[16];
}nidio96_private;
#define devpriv ((nidio96_private *)dev->private)

static int nidio96_find_device(comedi_device *dev,comedi_devconfig *it);
static int setup_mite(comedi_device *dev);


static int nidio96_8255_cb(int dir,int port,int data,void *arg)
{
	int iobase=(int)arg;

	if(dir){
		writeb(data,iobase+port);
		return 0;
	}else{
		return readb(iobase+port);
	}
}

int nidio96_init(comedi_device *dev,comedi_devconfig *it)
{
	comedi_subdevice *s;
	int ret;
	
	if(strcmp("pci-dio-96",it->board_name))
		return 0;
	
	printk("comedi%d: pci-dio: 0x%04x",dev->minor,dev->iobase);

	dev->driver_name="nidio96";
	dev->board_name="pci-dio-96";
	dev->irq=0;
	
	dev->n_subdevices=4;
	if((ret=alloc_subdevices(dev))<0)
		return ret;
	if((ret=alloc_private(dev,sizeof(nidio96_private)))<0)
		return ret;
	
	ret=nidio96_find_device(dev,it);
	if(ret<0)goto out;

	ret=setup_mite(dev);
	if(ret<0)goto out;
	
	s=dev->subdevices+0;

	subdev_8255_init(dev,dev->subdevices+0,nidio96_8255_cb,(void *)(dev->iobase+NIDIO_A));
	subdev_8255_init(dev,dev->subdevices+1,nidio96_8255_cb,(void *)(dev->iobase+NIDIO_B));
	subdev_8255_init(dev,dev->subdevices+2,nidio96_8255_cb,(void *)(dev->iobase+NIDIO_C));
	subdev_8255_init(dev,dev->subdevices+3,nidio96_8255_cb,(void *)(dev->iobase+NIDIO_D));

	dev->rem=nidio96_rem;
	
	printk("\n");

	return 1;
out:
	nidio96_rem(dev);

	return ret;
}

static int nidio96_rem(comedi_device *dev)
{
	if(devpriv->mite_io_addr){
		iounmap(devpriv->mite_io_addr);
		devpriv->mite_io_addr=NULL;
	}
	if(devpriv->daq_io_addr){
		iounmap(devpriv->daq_io_addr);
		devpriv->daq_io_addr=NULL;
	}

	return 0;
}


static int nidio96_find_device(comedi_device *dev,comedi_devconfig *it)
{
	struct pci_dev *pci_device=NULL;
	
	if(!pci_present()){
		printk(" no PCI bus\n");
		return -EINVAL;
	}

	for(pci_device=pci_devices;pci_device;pci_device=pci_device->next){
		if(pci_device->vendor==PCI_VENDOR_ID_NATINST){
			if(pci_device->device==PCI_DEVICE_ID_DIO96)
				break;
			printk("unknown NI device found with device_id=0x%04x\n",
				pci_device->device);
		}
	}
	if(!pci_device){
		printk("no device found\n");
		return -EINVAL;
	}

	devpriv->pci_device=pci_device;

	return 0;
}


/* should merge this with PCI-MIO mite setup. */

static int setup_mite(comedi_device *dev)
{
	unsigned long			offset;
	u32				addr;

	addr=devpriv->pci_device->base_address[1];
	devpriv->daq_phys_addr=addr;
	offset = devpriv->daq_phys_addr & ~PAGE_MASK;
	devpriv->daq_io_addr = ioremap(devpriv->daq_phys_addr & PAGE_MASK, PCI_DIO_SIZE + offset )
		+ offset;
#ifdef PCI_DEBUG
	printk("DAQ:0x%08lx mapped to %p, ",devpriv->daq_phys_addr,devpriv->daq_io_addr);
#endif

	addr=devpriv->pci_device->base_address[0];
	devpriv->mite_phys_addr=addr;
	offset = devpriv->mite_phys_addr & ~PAGE_MASK;
	devpriv->mite_io_addr = ioremap(devpriv->mite_phys_addr & PAGE_MASK, PCI_MITE_SIZE + offset )
		+ offset;
#ifdef PCI_DEBUG
	printk("MITE:0x%08lx mapped to %p ",devpriv->mite_phys_addr,devpriv->mite_io_addr);
#endif
	
	/* XXX don't know what the 0xc0 and 0x80 mean */
	writel(devpriv->daq_phys_addr | 0x80 , devpriv->mite_io_addr + 0xc0 );
	
	dev->iobase = (int) devpriv->daq_io_addr;
	
	return 0;
}

